package com.ubsidi.epos_2021.adapters;

import android.content.Context;
import android.graphics.Color;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.AppCompatImageView;
import androidx.core.content.ContextCompat;
import androidx.recyclerview.widget.RecyclerView;

import com.bumptech.glide.Glide;
import com.google.android.material.card.MaterialCardView;
import com.ubsidi.R;
import com.ubsidi.epos_2021.MyApp;
import com.ubsidi.epos_2021.comman.Validators;
import com.ubsidi.epos_2021.interfaces.RecyclerviewItemClickListener;
import com.ubsidi.epos_2021.models.Product;
import com.ubsidi.epos_2021.models.SiteSetting;

import java.io.File;
import java.util.ArrayList;

public class ProductsAdapter extends RecyclerView.Adapter<ProductsAdapter.ViewHolder> {
    private final ArrayList<Product> products;
    private final RecyclerviewItemClickListener itemClickListener;
    private final MyApp myApp = MyApp.getInstance();
    private boolean showImages;
    private boolean showBackground = false;

    public ProductsAdapter(ArrayList<Product> products, RecyclerviewItemClickListener itemClickListener) {
        this.products = products;
        this.itemClickListener = itemClickListener;
        showImages = myApp.productImageEnabled();
        SiteSetting productBackgroundMode = myApp.findSetting("product_background_type");
        showBackground = productBackgroundMode != null && productBackgroundMode.value.equalsIgnoreCase("full");
    }

    @NonNull
    @Override
    public ProductsAdapter.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new ViewHolder(LayoutInflater.from(parent.getContext()).inflate(R.layout.item_product, parent, false));
    }

    @Override
    public void onBindViewHolder(@NonNull ProductsAdapter.ViewHolder holder, int position) {
        try {
            Product product = products.get(position);
            if (product.name!=null) {
                holder.tvProduct.setText(product.name);
            }
            holder.tvPrice.setText(MyApp.currencySymbol + MyApp.df.format(product.selected_price));
            Context context = holder.itemView.getContext();
            if (showImages && !Validators.isNullOrEmpty(product.image)) {
                Glide.with(context)
                        .load(new File(context.getExternalFilesDir("epos2021").getAbsolutePath()) + "/" + product.id + ".png")
                        .into(holder.ivProduct);
                holder.ivProduct.setVisibility(View.VISIBLE);
            } else {
                holder.ivProduct.setVisibility(View.GONE);
            }

            holder.cvProduct.setOnClickListener(v -> {
                if (itemClickListener != null) {
                    itemClickListener.onItemClick(position, product);
                }
            });
            try {

                if (Validators.isNullOrEmpty(product.top_color) || !showBackground) {
                    holder.cvProduct.setCardBackgroundColor(ContextCompat.getColor(context, R.color.white));
                    holder.tvProduct.setTextColor(ContextCompat.getColor(context, R.color.black));
                    holder.tvPrice.setTextColor(ContextCompat.getColor(context, R.color.chip_green_color));
                } else {
                    holder.cvProduct.setCardBackgroundColor(Color.parseColor(product.top_color));
                    if (Validators.isNullOrEmpty(product.bottom_color)) {
                        holder.tvProduct.setTextColor(Color.parseColor("#ffffff"));
                        holder.tvPrice.setTextColor(Color.parseColor("#ffffff"));
                    } else {
                        holder.tvProduct.setTextColor(Color.parseColor(product.bottom_color));
                        holder.tvPrice.setTextColor(Color.parseColor(product.bottom_color));
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            int count = myApp.getCartProductsCount(product.id);
            holder.tvCount.setText("" + count);
            if (count > 0) {
                holder.tvCount.setVisibility(View.VISIBLE);
            } else {
                holder.tvCount.setVisibility(View.GONE);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public int getItemCount() {
        return products.size();
    }

    protected static class ViewHolder extends RecyclerView.ViewHolder {

        TextView tvProduct, tvCount, tvPrice;
        LinearLayout llItemBg, llProduct;
        AppCompatImageView ivProduct;
        private MaterialCardView cvProduct;

        public ViewHolder(@NonNull View itemView) {
            super(itemView);
            ivProduct = itemView.findViewById(R.id.ivProduct);
            tvProduct = itemView.findViewById(R.id.tvProduct);
            tvCount = itemView.findViewById(R.id.tvCount);
            tvPrice = itemView.findViewById(R.id.tvPrice);
            llItemBg = itemView.findViewById(R.id.llItemBg);
            llProduct = itemView.findViewById(R.id.llProduct);
            cvProduct = itemView.findViewById(R.id.cvProduct);

        }
    }
}
