package com.ubsidi.epos_2021.daos;

import androidx.room.Dao;
import androidx.room.Delete;
import androidx.room.Insert;
import androidx.room.OnConflictStrategy;
import androidx.room.Query;
import androidx.room.Transaction;
import androidx.room.Update;

import com.ubsidi.epos_2021.daos.relations.OrderWithItems;
import com.ubsidi.epos_2021.models.Order;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by Amrish on 19-05-2021.
 */
@Dao
public interface OrderDao {
    //If we want to include completed orders in it
    @Query("SELECT * FROM 'Order' WHERE order_type_id=:orderTypeId AND " +
            "is_archived=:isArchived AND " +
            "(" +
            "(date(created_at) >=:startDate  AND order_status_id IN (:completedIds))" +
            " OR " +
            "order_status_id IN (:pendingIds)" +
            ") " +
            "ORDER BY _id DESC")
    List<Order> allPendingOrders(String orderTypeId, String startDate, List<String> pendingIds, List<String> completedIds, boolean isArchived);

    //If we do not want to include completed orders in it
    @Query("SELECT * FROM 'Order' WHERE order_type_id=:orderTypeId AND " +
            "is_archived=:isArchived AND " +
//            "(date(created_at) >=:startDate AND order_status_id IN (:pendingIds)) OR " +
            "order_status_id IN (:pendingIds)" +
            "ORDER BY _id DESC")
    List<Order> allPendingOrders(String orderTypeId, List<String> pendingIds, boolean isArchived);

    //Show only pending orders
    @Query("SELECT * FROM 'Order' WHERE order_type_id=:orderTypeId AND is_archived=:isArchived AND order_status_id IN (:status_ids) ORDER BY _id DESC")
    List<Order> inprogress(String orderTypeId, List<String> status_ids, boolean isArchived);

    //Show only completed orders after some date
    @Query("SELECT * FROM 'Order' WHERE order_type_id=:orderTypeId AND is_archived=:isArchived AND date(created_at)>=:startDate AND order_status_id IN (:status_ids) ORDER BY _id DESC")
    List<Order> completed(String orderTypeId, String startDate, List<String> status_ids, boolean isArchived);

    //Show only completed orders after some date
    //@Query("SELECT * FROM 'Order' WHERE is_uploaded_on_server=0")
    @Query("SELECT count() as order_count FROM 'Order' WHERE is_uploaded_on_server=0")
    Integer failedOrder();

    @Query("SELECT * FROM 'Order' WHERE is_uploaded_on_server=0")
    List<Order> failedOrderList();

    @Query("SELECT * FROM 'Order' WHERE  order_status_id IN (:status_ids) AND is_archived=:isArchived AND date(created_at) >=:startDate AND date(created_at)<=:endDate AND order_type_id!=3 ORDER BY _id DESC")
    List<Order> historyListWithoutDelivery(String startDate, String endDate, List<String> status_ids, boolean isArchived);

    @Query("SELECT * FROM 'Order' WHERE  order_status_id IN (:status_ids) AND is_archived=:isArchived AND date(created_at) >=:startDate AND date(created_at)<=:endDate AND order_type_id==3 ORDER BY _id DESC")
    List<Order> historyListWithDelivery(String startDate, String endDate, List<String> status_ids, boolean isArchived);

    @Query("SELECT * FROM 'Order' WHERE  order_status_id IN (:status_ids)")
    List<Order> getRunningOrder(List<String> status_ids);

    @Query("SELECT * FROM 'Order' WHERE is_archived=:isArchived AND order_status_id IN (:status_ids) AND date(created_at) >=:startDate AND date(created_at)<=:endDate AND (id=:query OR _id=:query OR customer_name LIKE :query) ORDER BY _id DESC")
    List<Order> historyList(String startDate, String endDate, List<String> status_ids, String query, boolean isArchived);

    @Query("SELECT 'Order'.* FROM 'Order' INNER JOIN Customer ON `Order`._customer_id=Customer._id WHERE `Order`.is_archived=:isArchived AND `Order`.order_status_id IN (:status_ids) AND date(`Order`.created_at) >=:startDate AND date(`Order`.created_at)<=:endDate AND (`Order`.id=:query OR `Order`.id=:query OR `Order`.customer_name LIKE :query OR Customer.mobile LIKE :query) ORDER BY `Order`.id Desc")
    List<Order> historyListCustomer(String startDate, String endDate, List<String> status_ids, String query, boolean isArchived);

    @Query("SELECT * FROM 'Order' WHERE is_archived=:isArchived AND order_status_id IN (:status_ids) AND date(created_at) >=:startDate AND date(created_at)<=:endDate AND (table_id=:query) ORDER BY _id DESC")
    List<Order> historyListFilterTableWise(String startDate, String endDate, List<String> status_ids, String query, boolean isArchived);

    @Query("SELECT * FROM 'Order' WHERE _id=:orderId")
    Order view(int orderId);

    @Query("SELECT * FROM 'Order' WHERE table_id=:tableId AND order_type_id='1' AND order_status_id NOT IN ('5','10') ORDER BY _id DESC LIMIT 1")
    Order findLastOrderOfTable(String tableId);

    @Query("SELECT * FROM 'Order' WHERE id=:orderId ORDER BY _id DESC LIMIT 1")
    Order view(String orderId);

    @Query("SELECT * FROM 'Order' WHERE unique_id=:uniqueId ORDER BY _id DESC LIMIT 1")
    Order viewByUniqueId(String uniqueId);

    @Query("SELECT * FROM 'Order'  ORDER BY datetime(updated_at) DESC LIMIT 1")
    Order lastUpdatedOrder();

    @Query("SELECT * FROM 'Order' WHERE order_type_id=:order_type_id ORDER BY datetime(updated_at) DESC LIMIT 1")
    Order lastUpdatedOrder(String order_type_id);

    @Delete
    void delete(Order order);

    @Query("DELETE FROM `Order` WHERE id IN (:ids)")
    void deleteMultiple(List<String> ids);

    @Query("DELETE FROM `Order` WHERE id NOT IN(:ids) AND order_type_id=:orderTypeId AND is_archived=:isArchived AND order_status_id IS NOT '5' AND order_status_id IS NOT '10'")
    void deleteOtherIdsButNotThis(String orderTypeId, List<String> ids, boolean isArchived);

    @Update
    int update(Order order);

    @Update(onConflict = OnConflictStrategy.REPLACE)
    int updateReplace(Order order);

    @Insert
    long insertWOReplace(Order order);

    @Query("UPDATE `Order` SET total_paid=(SELECT SUM(amount) FROM OrderPayment WHERE _order_id=:orderId) WHERE _id=:orderId")
    void updatePaidAmount(int orderId);

    @Query("UPDATE `Order` set customer_id=:customer_id WHERE _customer_id=:customerId")
    void updateCustomerId(String customer_id, int customerId);

    @Query("UPDATE `Order` set display_order_id=:displayOrderId WHERE _id=:orderId")
    void updateDisplayOrderId(int orderId, String displayOrderId);

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    long insert(Order order);

    @Query("SELECT SUM(total) as total FROM OrderItem WHERE _order_id=:orderId and is_delete=0")
    float getItemSubTotal(int orderId);

    @Query("SELECT SUM(total) as total FROM OrderItem WHERE order_id=:orderId")
    float getItemSubTotal(String orderId);

    @Transaction
    @Query("SELECT * FROM `Order` WHERE id=:orderid")
    OrderWithItems orderWithItems(String orderid);

    @Transaction
    @Query("SELECT * FROM `Order`")
    List<OrderWithItems> ordersWithItems();

    @Transaction
    @Query("SELECT * FROM `Order` WHERE _id=:orderid")
    OrderWithItems orderWithItems(int orderid);


    @Query("SELECT COUNT(*) FROM `Order` WHERE _customer_id=:customerId")
    int customersTotalOrder(int customerId);

    @Query("DELETE FROM `Order`")
    void nukeOrders();

    @Query("DELETE FROM `sqlite_sequence` WHERE `name` = 'Customer' OR `name` = 'Order' OR name='OrderItem' OR name='OrderItemAddon' OR name='OrderItemIngredient' OR name='OrderPayment' OR name='OrderSplit' OR name='User' OR name='Reservation' OR name='SiteSetting' OR name='Voucher'")
    void resetPrimaryKey();

    @Query("DELETE FROM `sqlite_sequence` WHERE `name` = 'Order' OR name='OrderItem' OR name='OrderItemAddon' OR name='OrderItemIngredient' OR name='OrderPayment' OR name='OrderSplit'")
    void resetOrderRelatedPrimaryKey();

    @Query("UPDATE `Order` SET is_archived=:isArchived WHERE _id IN (:selectedOrderIds)")
    void archiveStatus(ArrayList<Integer> selectedOrderIds, boolean isArchived);

    @Query("SELECT `Order`.id FROM `Order`  WHERE _id IN (:selectedOrderIds) AND id IS NOT NULL")
    List<String> onlineIds(ArrayList<Integer> selectedOrderIds);

    @Query("SELECT COUNT(*) FROM `Order` WHERE order_type_id=:orderTypeId AND order_status_id IS NOT 5 AND order_status_id IS NOT 10")
    int activeOrdersCount(String orderTypeId);

    @Query("SELECT COUNT(*) FROM `Order` WHERE order_status_id IS NOT 5 AND order_status_id IS NOT 10")
    int activeOrdersCount();

    @Query("SELECT * FROM `Order` WHERE order_status_id IS NOT 5 AND order_status_id IS NOT 10")
    List<Order> activeOrders();

    @Query("SELECT * FROM `Order` WHERE id IN (:orderIds) AND id IS NOT NULL")
    List<Order> listWithOnlineIds(ArrayList<String> orderIds);

    @Query("UPDATE `Order` SET order_status_id=:statusId  WHERE _id IN (:selectedOrderIds)")
    void updateOrderStatusInBulk(ArrayList<Integer> selectedOrderIds, String statusId);
}
