package com.ubsidi.epos_2021.fragment;

import android.location.Location;
import android.os.AsyncTask;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.fragment.app.DialogFragment;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.ParsedRequestListener;
import com.google.android.material.button.MaterialButton;
import com.google.android.material.chip.Chip;
import com.google.android.material.textfield.MaterialAutoCompleteTextView;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.ubsidi.R;
import com.ubsidi.epos_2021.MyApp;
import com.ubsidi.epos_2021.adapters.CustomerAutoSuggestAdapter;
import com.ubsidi.epos_2021.comman.CommonFunctions;
import com.ubsidi.epos_2021.comman.Validators;
import com.ubsidi.epos_2021.daos.AppDatabase;
import com.ubsidi.epos_2021.interfaces.DialogDismissListener;
import com.ubsidi.epos_2021.models.ApiError;
import com.ubsidi.epos_2021.models.Customer;
import com.ubsidi.epos_2021.models.Postcode;
import com.ubsidi.epos_2021.models.SiteSetting;
import com.ubsidi.epos_2021.network.ApiEndPoints;
import com.ubsidi.epos_2021.utils.LogUtils;
import com.ubsidi.epos_2021.utils.ToastUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by Amrish on 05-05-2021.
 */
public class CustomerInfoDialogFragment extends DialogFragment {
    @Override
    public int getTheme() {
        return R.style.MyDialog;
    }

    @Override
    public void onStart() {
        super.onStart();
        getDialog().setCancelable(false);
        getDialog().getWindow().setLayout(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
    }

    private DialogDismissListener dialogDismissListener;
    public void setDialogDismissListener(DialogDismissListener dialogDismissListener) {
        this.dialogDismissListener = dialogDismissListener;
    }

    private MaterialButton btnConfirm, btnCancel;
    private MaterialButton btnSearch;
    private String orderTypeId;
    private AlertDialog progressBarDialog;
    private EditText etPostcode, etMobile, etEmail;
    private MaterialAutoCompleteTextView etName;
    private TextView tvTitle, tvDistance;
    private Postcode searchedPostcode;
    private ArrayList<Customer> customers = new ArrayList<>();
    private CustomerAutoSuggestAdapter customerAutoSuggestAdapter;
    private Customer selectedCustomer;
    private AppDatabase appDatabase;
    private LinearLayout llDeliveryFields, llGuest;
    private EditText etHouseNo, etStreet, etCity;
    private SiteSetting findlatitude, findLongitude, lookupSetting;
    private MyApp myApp = MyApp.getInstance();
    private ImageView ivSecDecrease, ivSecIncrease;
    private TextView tvModifyQty, tvTableNo;
    private float distance;
    private boolean edit;
    private int guestCount = 1;
    private Chip chipBack;

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_customer_info, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        try {
            appDatabase = MyApp.getInstance().appDatabase;
            if (getArguments() != null) {
                selectedCustomer = new Gson().fromJson(getArguments().getString("customer"), Customer.class);
                orderTypeId = getArguments().getString("order_type_id");
                edit = getArguments().getBoolean("edit");
            }
            findLongitude = myApp.findSetting("address_longitude");
            findlatitude = myApp.findSetting("address_latitude");
            lookupSetting = myApp.findSetting("postcode_lookup_visible");
            initViews(view);
            updateView();
            setListeners();
            if (!edit)
                fetchCustomers();
        } catch (JsonSyntaxException e) {
            e.printStackTrace();
        }

    }

    private void updateView() {
        try {
            if (selectedCustomer != null) {
                tvTitle.setText("Customer Info");
                etName.setText(selectedCustomer.name);
                etMobile.setText(selectedCustomer.mobile);
                etEmail.setText(selectedCustomer.email);
                etPostcode.setText(selectedCustomer.postcode);
                etStreet.setText(selectedCustomer.street);
                etCity.setText(selectedCustomer.city);
                etHouseNo.setText(selectedCustomer.house_no);
                if (!Validators.isNullOrEmpty(selectedCustomer.distance)) {
                    distance = Float.parseFloat(selectedCustomer.distance);
                    tvDistance.setText(distance + " Miles Away");
                    tvDistance.setVisibility(View.VISIBLE);
                }
                if (!Validators.isNullOrEmpty(orderTypeId) && orderTypeId.equalsIgnoreCase("1")) {
                    llGuest.setVisibility(View.VISIBLE);
                    if (selectedCustomer != null) {
                        tvModifyQty.setText(selectedCustomer.no_guest + "");
                        if (selectedCustomer.selectedTable != null) {
                            tvTableNo.setText(selectedCustomer.selectedTable.number);
                        }
                        guestCount = selectedCustomer.no_guest;
                    }
                    tvTableNo.setVisibility(View.VISIBLE);
                } else {
                    tvTableNo.setVisibility(View.GONE);
                    llGuest.setVisibility(View.GONE);
                }
            } else {
                tvTitle.setText("New Order");
            }
            if (lookupSetting != null && lookupSetting.value.equalsIgnoreCase("yes")) {
                btnSearch.setVisibility(View.VISIBLE);
            } else {
                btnSearch.setVisibility(View.GONE);
            }
        } catch (NumberFormatException e) {
            e.printStackTrace();
        }
    }

    private void setListeners() {
        //1 = Dine In, 2= Collection, 3 = Delivery, 4 = Bar, 5= Waiting, 6 Web Order

        try {
            chipBack.setOnClickListener(view -> {
                CommonFunctions.hideKeyboard(requireActivity(), btnCancel);
                dismiss();
            });
            btnCancel.setOnClickListener(view -> {
                CommonFunctions.hideKeyboard(requireActivity(), btnCancel);
                dismiss();
            });
            btnConfirm.setOnClickListener(view -> {
                if (isValid()) {
                    sendCustomerDetails();
                }
            });
            btnSearch.setOnClickListener(view -> {
                if (Validators.isNullOrEmpty(etPostcode.getText().toString())) {
                    etPostcode.setError("Please enter postcode");
                    etPostcode.requestFocus();
                } else {
                    searchPostcode(etPostcode.getText().toString());
                }
            });
            etName.setOnItemClickListener((adapterView, view, i, l) -> {
                int indext = customers.indexOf(adapterView.getItemAtPosition(i));
                if (indext != -1) {
                    selectedCustomer = customers.get(indext);
                    updateView();
                }

            });
            ivSecIncrease.setOnClickListener(view -> {
                guestCount++;
                tvModifyQty.setText("" + guestCount);
            });

            ivSecDecrease.setOnClickListener(view -> {
                if (guestCount > 1) {
                    guestCount--;
                    tvModifyQty.setText("" + guestCount);
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void sendCustomerDetails() {
        try {
            selectedCustomer.name = etName.getText().toString();
            selectedCustomer.mobile = etMobile.getText().toString();
            selectedCustomer.email = etEmail.getText().toString();

            selectedCustomer.postcode = etPostcode.getText().toString();
            selectedCustomer.street = etStreet.getText().toString();
            selectedCustomer.city = etCity.getText().toString();
            selectedCustomer.house_no = etHouseNo.getText().toString();
            selectedCustomer.distance = MyApp.df.format(distance);
            if (searchedPostcode != null) {
                selectedCustomer.state = searchedPostcode.dependent_locality;
            }
            selectedCustomer.country = "United Kingdom";
            selectedCustomer.no_guest = guestCount;
            if (dialogDismissListener != null) {
                dialogDismissListener.onDialogDismiss(selectedCustomer);
                dismiss();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private boolean isValid() {

        try {
            if (!Validators.isNullOrEmpty(orderTypeId) && orderTypeId.equalsIgnoreCase("3")) {
                if (Validators.isNullOrEmpty(etName.getText().toString())) {
                    etName.setError("Please enter customer name");
                    etName.requestFocus();
                    return false;
                }

                if (Validators.isNullOrEmpty(etMobile.getText().toString())) {
                    etMobile.setError("Please enter mobile number");
                    etMobile.requestFocus();
                    return false;
                }
                if (Validators.isNullOrEmpty(etPostcode.getText().toString())) {
                    etPostcode.setError("Please lookup postcode");
                    etPostcode.requestFocus();
                    return false;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    private void initViews(View view) {
        try {
            progressBarDialog = CommonFunctions.customProgressDialog(getActivity());
            chipBack = view.findViewById(R.id.chipBack);
            btnCancel = view.findViewById(R.id.btnCancel);
            btnConfirm = view.findViewById(R.id.btnConfirm);
            tvTitle = view.findViewById(R.id.tvTitle);

            tvTableNo = view.findViewById(R.id.tvTableNo);
            llGuest = view.findViewById(R.id.llGuest);
            tvModifyQty = view.findViewById(R.id.tvModifyQty);
            ivSecIncrease = view.findViewById(R.id.ivSecIncrease);
            ivSecDecrease = view.findViewById(R.id.ivSecDecrease);

            etEmail = view.findViewById(R.id.etEmail);
            etMobile = view.findViewById(R.id.etPhone);
            etName = view.findViewById(R.id.etName);
            etPostcode = view.findViewById(R.id.etPostcode);

            tvDistance = view.findViewById(R.id.tvDistance);
            llDeliveryFields = view.findViewById(R.id.llDeliveryFields);

            etCity = view.findViewById(R.id.etTown);
            etHouseNo = view.findViewById(R.id.etHouse);
            etStreet = view.findViewById(R.id.etStreet);

            btnSearch = view.findViewById(R.id.btnSearch);

            getOfflineCustomer();

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void getOfflineCustomer() {
        new Thread(() -> {
            try {
                if (!customers.isEmpty()) {
                    customers.clear();
                }
                if (getActivity() != null) {
                    getActivity().runOnUiThread(() -> {
                        customers.addAll(appDatabase.customerDao().listNoNull());
                        LogUtils.e("Cusomer info getOfflineCustomer " + customers.size());
                        customerAutoSuggestAdapter = new CustomerAutoSuggestAdapter(getActivity(), customers);
                        etName.setAdapter(customerAutoSuggestAdapter);
                        LogUtils.e("Customer adapter set!");
                    });
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }).start();
    }

    private void searchPostcode(String postcode) {
        try {
            progressBarDialog.show();
            progressBarDialog.getWindow().setDimAmount(0.4f);
            AndroidNetworking.get(ApiEndPoints.postcode + postcode)
                    .build()
                    .getAsObject(Postcode.class, new ParsedRequestListener<Postcode>() {
                        @Override
                        public void onResponse(Postcode response) {
                            try {
                                progressBarDialog.dismiss();
                                searchedPostcode = response;
                                etStreet.setText(searchedPostcode.street);
                                etCity.setText(searchedPostcode.post_town);
                                CommonFunctions.hideKeyboard(getActivity(), etPostcode);
                                calculateDistance(searchedPostcode.latitude, searchedPostcode.longitude);
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                progressBarDialog.dismiss();
                                if (anError.getErrorCode() == 400) {
                                    ApiError apiError = anError.getErrorAsObject(ApiError.class);
                                    ToastUtils.makeSnackToast(getActivity(), apiError.getMessage());
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void calculateDistance(String latitude, String longitude) {
        try {
            if (!Validators.isNullOrEmpty(findlatitude.value) && !Validators.isNullOrEmpty(findLongitude.value) && !Validators.isNullOrEmpty(latitude) && !Validators.isNullOrEmpty(longitude)) {
                Location locationRestaurant = new Location("Restaurant Location");
                locationRestaurant.setLongitude(Double.parseDouble(findLongitude.value));
                locationRestaurant.setLatitude(Double.parseDouble(findlatitude.value));

                Location locationUser = new Location("User Location");
                locationUser.setLongitude(Double.parseDouble(longitude));
                locationUser.setLatitude(Double.parseDouble(latitude));

                float d = locationRestaurant.distanceTo(locationUser) * 0.000621371f;
                distance = Float.parseFloat(MyApp.df.format(d));
                tvDistance.setText(distance + " Miles Away");
                tvDistance.setVisibility(View.VISIBLE);
            }
        } catch (NumberFormatException e) {
            e.printStackTrace();
        }
    }

    private void fetchCustomers() {
        try {
            if (MyApp.getInstance().isConnected(getActivity()) && "auto".equalsIgnoreCase(myApp.myPreferences.getOrderSyncMode())) {
                fetchOnlineCustomers();
            } else {
                new DbSaverAsync().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, customers);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void fetchOnlineCustomers() {
        try {
            AndroidNetworking.get(ApiEndPoints.customers)
                    .addQueryParameter("nopaginate", "1")
                    .build()
                    .getAsObjectList(Customer.class, new ParsedRequestListener<List<Customer>>() {

                        @Override
                        public void onResponse(List<Customer> response) {
                            try {
                                new DbSaverAsync().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, (ArrayList<Customer>) response);
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                if (anError.getErrorCode() == 400) {
                                    ApiError apiError = anError.getErrorAsObject(ApiError.class);
                                    ToastUtils.makeSnackToast(getActivity(), apiError.getMessage());
                                }
                                new DbSaverAsync().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, customers);
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private class DbSaverAsync extends AsyncTask<ArrayList<Customer>, Void, String> {

        @Override
        protected void onPreExecute() {
            super.onPreExecute();
        }

        @Override
        protected String doInBackground(ArrayList<Customer>... lists) {
            try {
                for (Customer customer : lists[0]) {
                    Customer dbCustomer = appDatabase.customerDao().view(customer.id);
                    if (dbCustomer == null) {
                        appDatabase.customerDao().insert(customer);
                    } else {
                        appDatabase.customerDao().update(customer);
                    }
                }
                if (getActivity() != null)
                    getActivity().runOnUiThread(() -> {
                        customers.clear();
                        customers.addAll(appDatabase.customerDao().listNoNull());
                    });
            } catch (Exception e) {
                e.printStackTrace();
            }
            return "Executed";
        }

        @Override
        protected void onPostExecute(String s) {
            super.onPostExecute(s);
            try {
                if (getActivity() != null) {
                    getActivity().runOnUiThread(() -> {
                        progressBarDialog.dismiss();
                        customerAutoSuggestAdapter.updateData(customers);
                    });
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }
}
