package com.ubsidi.epos_2021.online.services_receivers;

import android.app.job.JobParameters;
import android.app.job.JobService;
import android.content.Intent;
import android.util.Log;

import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.ParsedRequestListener;
import com.google.gson.Gson;
import com.ubsidi.epos_2021.MyApp;
import com.ubsidi.epos_2021.comman.Validators;
import com.ubsidi.epos_2021.models.ApiError;
import com.ubsidi.epos_2021.models.CallLogs;
import com.ubsidi.epos_2021.models.Reservation;
import com.ubsidi.epos_2021.models.Voucher;
import com.ubsidi.epos_2021.network.ApiEndPoints;
import com.ubsidi.epos_2021.online.models.Counts;
import com.ubsidi.epos_2021.utils.Constants;
import com.ubsidi.epos_2021.utils.LogUtils;

public class TiffintomPartnerJobService  extends JobService {

    private MyApp myApp = MyApp.getInstance();

    @Override
    public boolean onStartJob(JobParameters params) {
        MyApp.schedulerJob(getApplicationContext()); // reschedule the job
        Log.e("Job service","onStartJob");
        performBackgroundTask();
        return true;
    }

    @Override
    public boolean onStopJob(JobParameters params) {
        return true;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
    }

    private void performBackgroundTask()
    {
        try {
            if (myApp.isConnected(getApplicationContext())) {
                if (canWeUploadNextCallLog) {
                    new Thread(() -> {
                        CallLogs callLogs = myApp.appDatabase.calllogDao().uploadableFirst();
                        if (callLogs != null) {
                            uploadCallLog(callLogs);
                        }
                    }).start();
                }
                if (canWeUploadNextReservation) {
                    if (!myApp.isReservationOpen) {
                        new Thread(() -> {
                            Reservation reservation = myApp.appDatabase.reservationDao().uploadableFirst();
                            if (reservation != null) {
                                reservation.customer = myApp.appDatabase.customerDao().view(reservation._customer_id);
                                reservation.customer_id = reservation.customer.id;
                                uploadReservation(reservation);
                            } else {
                                LogUtils.e("No reservation found to upload");
                            }
                        }).start();
                    } else {
                        LogUtils.e("We cannot upload reservation as already on that screen");
                    }
                }
                if (canWeUploadNextVoucher) {
                    new Thread(() -> {
                        Voucher voucher = myApp.appDatabase.voucherDao().view(0);
                        if (voucher != null) {
                            voucher.customer = myApp.appDatabase.customerDao().view(voucher._customer_id);
                            voucher.customer_id = voucher.customer.id;
                            uploadVoucher(voucher);
                        }
                    }).start();
                }
                try {
                    if (!Validators.isNullOrEmpty(myApp.restaurant_id))
                        fetchCounts();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    boolean canWeUploadNextCallLog = true;
    boolean canWeUploadNextReservation = true;
    boolean canWeUploadNextVoucher = true;

    private void uploadCallLog(CallLogs callLogs) {
        try {
            canWeUploadNextCallLog = false;
            String url = ApiEndPoints.calllogs;
            if (!Validators.isNullOrEmpty(callLogs.id)) {
                url = url + callLogs.id;
            }
            AndroidNetworking.post(url)
                    .addBodyParameter(callLogs)
                    .build()
                    .getAsObject(CallLogs.class, new ParsedRequestListener<CallLogs>() {
                        @Override
                        public void onResponse(CallLogs response) {
                            canWeUploadNextCallLog = true;
                            LogUtils.e("Uploaded call log for number " + callLogs.number);
                            new Thread(() -> {
                                response._id = callLogs._id;
                                myApp.appDatabase.calllogDao().insert(response);
                            }).start();
                        }

                        @Override
                        public void onError(ANError anError) {
                            canWeUploadNextCallLog = true;
                            LogUtils.e("Cannot Upload call log for number " + callLogs.number);
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void uploadReservation(Reservation reservation) {
        try {
            canWeUploadNextReservation = false;
            String url = ApiEndPoints.reservations;
            if (!Validators.isNullOrEmpty(reservation.id)) {
                url = url + reservation.id;
            }
            LogUtils.e("Service Uploading reservation::" + new Gson().toJson(reservation));
            AndroidNetworking.post(url)
                    .addApplicationJsonBody(reservation)
                    .build()
                    .getAsObject(Reservation.class, new ParsedRequestListener<Reservation>() {
                        @Override
                        public void onResponse(Reservation response) {
                            canWeUploadNextReservation = true;
                            new Thread(() -> {
                                reservation.id = response.id;
                                reservation.uploadable = false;
                                reservation.customer_id = response.customer_id;
                                reservation.customer.id = response.customer.id;
                                myApp.appDatabase.customerDao().insert(reservation.customer);
                                myApp.appDatabase.reservationDao().insert(reservation);
                            }).start();
                        }

                        @Override
                        public void onError(ANError anError) {
                            LogUtils.e("Error in uploading reservation");
                            anError.printStackTrace();
                            if (anError.getErrorCode() == 400) {
                                if (anError.getErrorAsObject(ApiError.class).getCode().equalsIgnoreCase("404")) {
                                    new Thread(() -> {
                                        reservation.id = null;
                                        myApp.appDatabase.reservationDao().update(reservation);
                                        canWeUploadNextReservation = true;
                                    }).start();
                                } else {
                                    canWeUploadNextReservation = true;
                                }
                            } else {
                                canWeUploadNextReservation = true;
                            }

                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void uploadVoucher(Voucher voucher) {
        try {
            canWeUploadNextVoucher = false;
            String url = ApiEndPoints.voucher_add;
            if (!Validators.isNullOrEmpty(voucher.id)) {
                url = url + voucher.id;
            }
            LogUtils.e("Service Uploading voucher::" + new Gson().toJson(voucher));
            AndroidNetworking.post(url)
                    .addApplicationJsonBody(voucher)
                    .build()
                    .getAsObject(Voucher.class, new ParsedRequestListener<Voucher>() {
                        @Override
                        public void onResponse(Voucher response) {
                            canWeUploadNextReservation = true;
                            new Thread(() -> {
                                voucher.id = response.id;
                                //                            voucher.uploadable = false;
                                voucher.customer_id = response.customer_id;
                                voucher.customer.id = response.customer.id;
                                myApp.appDatabase.customerDao().insert(voucher.customer);
                                myApp.appDatabase.voucherDao().insert(voucher);
                            }).start();
                        }

                        @Override
                        public void onError(ANError anError) {
                            LogUtils.e("Error in uploading voucher");
                            anError.printStackTrace();
                            if (anError.getErrorCode() == 400) {
                                if (anError.getErrorAsObject(ApiError.class).getCode().equalsIgnoreCase("404")) {
                                    new Thread(() -> {
                                        voucher.id = null;
                                        myApp.appDatabase.voucherDao().insert(voucher);
                                        canWeUploadNextVoucher = true;
                                    }).start();
                                } else {
                                    canWeUploadNextVoucher = true;
                                }
                            } else {
                                canWeUploadNextVoucher = true;
                            }

                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    boolean lastBroadcastSent = false;

    public void fetchCounts() {
        try {
            if (Validators.isNullOrEmpty(myApp.restaurant_id)) return;
            AndroidNetworking.get(ApiEndPoints.restaurant_counts)
                    .addPathParameter("id", myApp.restaurant_id)
                    .build()
                    .getAsObject(Counts.class, new ParsedRequestListener<Counts>() {
                        @Override
                        public void onResponse(Counts response) {
                            if (response == null) return;
                            Intent pushNotification = new Intent(Constants.NOTIFICATION);
                            pushNotification.putExtra("refresh", true);
                            pushNotification.putExtra("counts", new Gson().toJson(response));

                            if (response.pending_orders > 0) {
                                MyApp.getInstance().startOrderSound();
                                pushNotification.putExtra("orders", true);
                                lastBroadcastSent = false;
                            } else {
                                MyApp.getInstance().stopOrderSound();
                                if (!lastBroadcastSent)
                                    pushNotification.putExtra("orders", true);
                                lastBroadcastSent = true;
                            }
                            if (response.pending_bookings > 0) {
                                pushNotification.putExtra("reservations", true);
                            }
                            if (response.new_reviews > 0) {
                                pushNotification.putExtra("reviews", true);
                            }
                            if (response.delay_orders > 0) {
                                lastBroadcastSent = false;
                                pushNotification.putExtra("orders", true);
                            } else {
                                if (!lastBroadcastSent)
                                    pushNotification.putExtra("orders", true);
                                lastBroadcastSent = true;
                            }
                            if (response.unread_chat > 0) {
                                pushNotification.putExtra("chat", true);
                            }
                            LocalBroadcastManager.getInstance(getApplicationContext()).sendBroadcast(pushNotification);
                        }

                        @Override
                        public void onError(ANError anError) {
                            anError.printStackTrace();
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
