package com.ubsidi.epos_2021.utils

import android.content.Context
import android.content.Intent
import android.database.Cursor
import android.net.Uri
import android.os.Environment
import android.os.Handler
import android.os.Looper
import android.os.Message
import android.util.Log
import androidx.annotation.NonNull
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.ubsidi.epos_2021.MyApp
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors


class DownloadManager {
    // Indicate that we would like to update download progress
    private val UPDATE_DOWNLOAD_PROGRESS = 1

    // Use a background thread to check the progress of downloading
    private val executor: ExecutorService = Executors.newFixedThreadPool(1)

    // Use a hander to update progress bar on the main thread
    private val mainHandler: Handler = Handler(
        Looper.getMainLooper()
    ) { msg ->
        if (msg.what == UPDATE_DOWNLOAD_PROGRESS) {
            val downloadProgress: Int = msg.arg1
            val intent = Intent(Constants.DOWNLOAD_PROGRESS)
            intent.putExtra(Constants.PROGRESS, downloadProgress)
            LocalBroadcastManager.getInstance(MyApp.getInstance()).sendBroadcast(intent)
            // Update your progress bar here.
            //progressBar.setProgress(downloadProgress)
        }
        true
    }

    fun downloadFile(
        context: Context,
        fileName: String,
        fileExtension: String,
        url: String?,
        destinationDirectory: String? = Environment.DIRECTORY_DOWNLOADS,
    ) {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val downloadManager =
                    context.getSystemService(Context.DOWNLOAD_SERVICE) as android.app.DownloadManager
                val uri: Uri = Uri.parse(url)
                val request: android.app.DownloadManager.Request =
                    android.app.DownloadManager.Request(uri)
                request.setNotificationVisibility(android.app.DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED)
                request.setDestinationInExternalPublicDir(
                    destinationDirectory,
                    fileName + fileExtension
                )
                val downloadId: Long = downloadManager.enqueue(request)
                var progress = 0
                var isDownloadFinished = false
                while (!isDownloadFinished) {
                    val cursor: Cursor = downloadManager.query(
                        android.app.DownloadManager.Query().setFilterById(downloadId)
                    )
                    if (cursor.moveToFirst()) {
                        val downloadStatus: Int =
                            cursor.getInt(cursor.getColumnIndex(android.app.DownloadManager.COLUMN_STATUS))
                        when (downloadStatus) {
                            android.app.DownloadManager.STATUS_RUNNING -> {
                                val totalBytes: Long =
                                    cursor.getLong(cursor.getColumnIndex(android.app.DownloadManager.COLUMN_TOTAL_SIZE_BYTES))
                                if (totalBytes > 0) {
                                    val downloadedBytes: Long =
                                        cursor.getLong(cursor.getColumnIndex(android.app.DownloadManager.COLUMN_BYTES_DOWNLOADED_SO_FAR))
                                    progress = (downloadedBytes * 100 / totalBytes).toInt()
                                }
                            }
                            android.app.DownloadManager.STATUS_SUCCESSFUL -> {
                                progress = 100
                                isDownloadFinished = true
                            }
                            android.app.DownloadManager.STATUS_PAUSED, android.app.DownloadManager.STATUS_PENDING -> {}
                            android.app.DownloadManager.STATUS_FAILED -> isDownloadFinished =
                                true
                        }
                        val message: Message = Message.obtain()
                        message.what = UPDATE_DOWNLOAD_PROGRESS
                        message.arg1 = progress
                        mainHandler.sendMessage(message)
                        if (progress >= 100) {
                            val intent = Intent(Constants.DOWNLOAD_PROGRESS)
                            intent.putExtra(Constants.PROGRESS, 100)
                            LocalBroadcastManager.getInstance(MyApp.getInstance())
                                .sendBroadcast(intent)
                            mainHandler.removeCallbacksAndMessages(null);
                        }
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    ToastUtils.makeToast(context, "Error is ${e.message} localizedMessage ${e.localizedMessage}")
                }
            }
        }
    }
}