package com.ubsidi.epos_2021.utils;

import android.annotation.TargetApi;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.net.NetworkRequest;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.lifecycle.LiveData;

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class NetworkWatcher extends LiveData<Boolean> {
    //    Variables
    private final Context context;

    private ConnectivityManager connectivityManager;
    private ConnectivityManager.NetworkCallback networkCallback;

    private NetworkRequest networkRequest;

    private NetworkStateWatcherReceiver networkStateWatcherReceiver;


    //    Constructors
    public NetworkWatcher(@NonNull Context context){
        this.context = context;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M){
            this.connectivityManager = context.getSystemService(ConnectivityManager.class);

            this.networkCallback = new ConnectivityManager.NetworkCallback(){
                @Override
                public void onLost(@NonNull Network network) {
                    NetworkWatcher.super.postValue(false);
                }

                @Override
                public void onCapabilitiesChanged(@NonNull Network network, @NonNull NetworkCapabilities networkCapabilities) {
                    NetworkWatcher.super.postValue(
                            networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET) &&
                                    networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
                    );
                }
            };
            this.networkRequest = new NetworkRequest.Builder()
                    .addTransportType(NetworkCapabilities.TRANSPORT_WIFI)
                    .addTransportType(NetworkCapabilities.TRANSPORT_CELLULAR)
                    .addTransportType(NetworkCapabilities.TRANSPORT_VPN)
                    .addTransportType(NetworkCapabilities.TRANSPORT_ETHERNET)
                    .addCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
                    .addCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
                    .build();
        } else {
            this.networkStateWatcherReceiver = new NetworkStateWatcherReceiver();
            this.networkStateWatcherReceiver.setOnNetworkChangedListener(NetworkWatcher.super::postValue);
        }
    }


    //    Override methods
    @Override
    protected void onActive() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M){
          /*  if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S){
                this.connectivityManager.registerBestMatchingNetworkCallback(this.networkRequest, this.networkCallback, null);
            } else */
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N){
                this.connectivityManager.registerDefaultNetworkCallback(this.networkCallback);
            } else {
                this.connectivityManager.registerNetworkCallback(this.networkRequest, this.networkCallback);
            }
        } else {
            this.context.registerReceiver(
                    this.networkStateWatcherReceiver,
                    new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION)
            );
        }
    }

    @Override
    protected void onInactive() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M){
            this.connectivityManager.unregisterNetworkCallback(this.networkCallback);
        } else {
            this.context.unregisterReceiver(this.networkStateWatcherReceiver);
        }
    }



    //    Inner method classes
    @TargetApi(Build.VERSION_CODES.LOLLIPOP_MR1)
    private static final class NetworkStateWatcherReceiver extends BroadcastReceiver {
        //        Variables
        private OnNetworkChangedListener onNetworkChangedListener;


        //        Constructors
        public NetworkStateWatcherReceiver() {
        }


        //        Override methods
        @Override
        public void onReceive(@NonNull Context context, @NonNull Intent intent) {
            if (this.onNetworkChangedListener != null) {
                boolean isConnected = this.isConnected(context);
                this.onNetworkChangedListener.onNetworkChangedListener(isConnected);
            }
        }


        //        Methods
        private boolean isConnected(@NonNull Context context){
            ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();

            return networkInfo != null && networkInfo.isConnected();
        }
        public void setOnNetworkChangedListener(@Nullable OnNetworkChangedListener onNetworkChangedListener) {
            this.onNetworkChangedListener = onNetworkChangedListener;
        }



        //        Inner interfaces
        private interface OnNetworkChangedListener{
            void onNetworkChangedListener(boolean isConnected);
        }
    }
}
