package com.tiffintom.partner1.adapters;

import android.app.Activity;
import android.content.res.ColorStateList;
import android.graphics.Paint;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.AppCompatImageView;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.RecyclerView;

import com.google.android.material.card.MaterialCardView;
import com.google.android.material.chip.Chip;
import com.tiffintom.partner1.MyApp;
import com.tiffintom.partner1.R;
import com.tiffintom.partner1.common.CommonFunctions;
import com.tiffintom.partner1.common.StickHeaderItemDecoration;
import com.tiffintom.partner1.common.Validators;
import com.tiffintom.partner1.interfaces.RecyclerItemViewClickListener;
import com.tiffintom.partner1.models.OrderStatusIndicatorHeader;
import com.tiffintom.partner1.models.RestaurantOrder;

import java.util.ArrayList;
import java.util.Date;

/**
 * Created by Amrish on 09-11-2020.
 */
public class OrdersAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder> implements StickHeaderItemDecoration.StickyHeaderInterface {
    private ArrayList<Object> ordersList;
    private Activity activity;
    private RecyclerItemViewClickListener itemViewClickListener, viewClickListener, printClickListener, rtcClicked;

    public OrdersAdapter(FragmentActivity activity, ArrayList<Object> ordersList, RecyclerItemViewClickListener itemViewClickListener, RecyclerItemViewClickListener viewClickListener, RecyclerItemViewClickListener printClickListener, RecyclerItemViewClickListener rtcClicked) {
        this.ordersList = ordersList;
        this.activity = activity;
        this.itemViewClickListener = itemViewClickListener;
        this.viewClickListener = viewClickListener;
        this.printClickListener = printClickListener;
        this.rtcClicked = rtcClicked;
    }

    public OrdersAdapter(FragmentActivity activity, ArrayList<Object> ordersList, RecyclerItemViewClickListener itemViewClickListener) {
        this.ordersList = ordersList;
        this.activity = activity;
        this.itemViewClickListener = itemViewClickListener;
    }

    @Override
    public int getItemViewType(int position) {
        if (ordersList.get(position) instanceof RestaurantOrder) {
            return R.layout.item_order_list;
        } else {
            return R.layout.item_orders_status_indicator;
        }
    }

    @NonNull
    @Override
    public RecyclerView.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View view = LayoutInflater.from(parent.getContext()).inflate(viewType, parent, false);
        if (viewType == R.layout.item_order_list) {
            return new OrderViewHolder(view);
        } else {
            return new OrderStatusHeaderViewHolder(view);
        }
    }

    @Override
    public void onBindViewHolder(@NonNull RecyclerView.ViewHolder holder, int position) {
        if (getItemViewType(position) == R.layout.item_orders_status_indicator) {
            OrderStatusHeaderViewHolder viewHolder = (OrderStatusHeaderViewHolder) holder;
            OrderStatusIndicatorHeader header = (OrderStatusIndicatorHeader) ordersList.get(position);
            viewHolder.tvTitle.setText(header.title);
        }
        if (getItemViewType(position) == R.layout.item_order_list) {
            OrderViewHolder viewHolder = (OrderViewHolder) holder;
            RestaurantOrder order = (RestaurantOrder) ordersList.get(position);
            boolean isDinein = Validators.isNullOrEmpty(order.order_type);
            boolean isPreOrder = false;
            boolean delayed = false;

            if (!isDinein) {
                if (order.delivery_date!=null) {
                    isPreOrder = CommonFunctions.isPreOrderNew(order.delivery_date, order.status);
                }
            }
            String paymentStatus;
            if (order.payment_status.equalsIgnoreCase("p")) {
                paymentStatus = "Paid";
            } else {
                paymentStatus = "Unpaid";
            }

            String chipMessage = "View";
            String addressString = "";
            String timeString = "";
            viewHolder.chipRtc.setVisibility(View.GONE);
            if (order.status.equalsIgnoreCase("pending")) {
                chipMessage = "View";
            } else if (order.status.equalsIgnoreCase("delivered")) {
                if (isDinein) {
                    chipMessage = "Served";
                } else {
                    if (order.order_type.equalsIgnoreCase("delivery")) {
                        chipMessage = "Delivered";
                    } else {
                        chipMessage = "Picked up";
                    }
                }
            } else if (order.status.equalsIgnoreCase("accepted") || order.status.equalsIgnoreCase("waiting") || order.status.equalsIgnoreCase("driver accepted")) {
                if (isDinein) {
                    chipMessage = "Ready to serve";
                } else {
                    if (order.order_type.equalsIgnoreCase("delivery")) {
                        chipMessage = "Mark as delivered";
                    } else {
                        chipMessage = "Mark as picked up";
                        if (MyApp.getInstance().getMyPreferences().getLoggedInRestaurant().RTC.equalsIgnoreCase("yes")) {
                            viewHolder.chipRtc.setVisibility(View.VISIBLE);
                        }
                    }
                    Date preparationDate = CommonFunctions.convertStringDateToDate(order.preparation, "hh:mm a");
                    Date nowDate = CommonFunctions.convertStringDateToDate(CommonFunctions.getCurrentTimeFormatted("hh:mm a"), "hh:mm a");

                    delayed = preparationDate.getTime() < nowDate.getTime();
                }

            } else if (order.status.equalsIgnoreCase("failed")) {
                chipMessage = "Rejected";
            }

            if (order.status.equalsIgnoreCase("pending")) {
                if (isPreOrder) {
                    viewHolder.cvOrder.setCardBackgroundColor(ContextCompat.getColor(activity, R.color.orders_card_bg_orange));
                } else {
                    viewHolder.cvOrder.setCardBackgroundColor(ContextCompat.getColor(activity, R.color.orders_card_bg_green));
                }
                viewHolder.tvOrderStatus.setTextColor(activity.getResources().getColor(R.color.white));
                viewHolder.tvCustomerName.setTextColor(activity.getResources().getColor(R.color.white));
                viewHolder.tvAddress.setTextColor(activity.getResources().getColor(R.color.white));
                viewHolder.tvTime.setTextColor(activity.getResources().getColor(R.color.white));
                viewHolder.tvOrderId.setTextColor(activity.getResources().getColor(R.color.white));
                viewHolder.tvView.setVisibility(View.INVISIBLE);
                viewHolder.ivPrint.setImageTintList(ColorStateList.valueOf(ContextCompat.getColor(activity, R.color.white)));

            } else {
                if (delayed) {
                    viewHolder.cvOrder.setCardBackgroundColor(ContextCompat.getColor(activity, R.color.orders_card_bg_delay));
                    viewHolder.tvOrderStatus.setTextColor(activity.getResources().getColor(R.color.white));
                    viewHolder.tvCustomerName.setTextColor(activity.getResources().getColor(R.color.white));
                    viewHolder.tvAddress.setTextColor(activity.getResources().getColor(R.color.white));
                    viewHolder.tvTime.setTextColor(activity.getResources().getColor(R.color.white));
                    viewHolder.tvView.setTextColor(activity.getResources().getColor(R.color.white));
                    viewHolder.tvOrderId.setTextColor(activity.getResources().getColor(R.color.white));
                    viewHolder.ivPrint.setImageTintList(ColorStateList.valueOf(ContextCompat.getColor(activity, R.color.white)));
                } else {
                    viewHolder.cvOrder.setCardBackgroundColor(ContextCompat.getColor(activity, R.color.orders_card_bg_grey));
                    viewHolder.tvOrderStatus.setTextColor(activity.getResources().getColor(R.color.light_black));
                    viewHolder.tvCustomerName.setTextColor(activity.getResources().getColor(R.color.light_black));
                    viewHolder.tvAddress.setTextColor(activity.getResources().getColor(R.color.light_black));
                    viewHolder.tvTime.setTextColor(activity.getResources().getColor(R.color.light_black));
                    viewHolder.tvView.setTextColor(activity.getResources().getColor(R.color.light_black));
                    viewHolder.tvOrderId.setTextColor(activity.getResources().getColor(R.color.light_black));
                    viewHolder.ivPrint.setImageTintList(ColorStateList.valueOf(ContextCompat.getColor(activity, R.color.black)));
                }
                viewHolder.tvView.setPaintFlags(Paint.UNDERLINE_TEXT_FLAG);
                viewHolder.tvView.setVisibility(View.VISIBLE);
            }
            if (order.status.equalsIgnoreCase("pending"))
                viewHolder.ivPrint.setVisibility(View.INVISIBLE);
            else {
                viewHolder.ivPrint.setVisibility(View.VISIBLE);
            }

            if (order.status.equalsIgnoreCase("delivered")) {
                viewHolder.chip.setEnabled(false);
                viewHolder.chip.setCloseIconVisible(true);
                viewHolder.chip.setCloseIcon(ContextCompat.getDrawable(activity, R.drawable.ic_baseline_green_done_24));
                viewHolder.chip.setCloseIconTint(ColorStateList.valueOf(ContextCompat.getColor(activity, R.color.chip_green_color)));
                viewHolder.chip.setTextColor(activity.getResources().getColor(R.color.chip_green_color));
                viewHolder.chip.setChipBackgroundColor(ContextCompat.getColorStateList(activity, R.color.orders_card_bg_grey));
            } else if (order.status.equalsIgnoreCase("failed")) {
                viewHolder.chip.setEnabled(false);
                viewHolder.chip.setCloseIconVisible(true);
                viewHolder.chip.setCloseIcon(ContextCompat.getDrawable(activity, R.drawable.ic_baseline_close_24));
                viewHolder.chip.setCloseIconTint(ColorStateList.valueOf(ContextCompat.getColor(activity, R.color.chip_orange_color)));
                viewHolder.chip.setTextColor(activity.getResources().getColor(R.color.chip_orange_color));
                viewHolder.chip.setChipBackgroundColor(ContextCompat.getColorStateList(activity, R.color.orders_card_bg_grey));
            } else {
                viewHolder.chip.setEnabled(true);
                viewHolder.chip.setCloseIconVisible(false);
                viewHolder.chip.setTextColor(activity.getResources().getColor(R.color.white));
                if (isPreOrder) {
                    viewHolder.chip.setChipBackgroundColor(ContextCompat.getColorStateList(activity, R.color.chip_orange_color));
                } else {
                    if (delayed) {
                        viewHolder.chip.setChipBackgroundColor(ContextCompat.getColorStateList(activity, R.color.delay_chip_orange_color));
                    } else {
                        viewHolder.chip.setChipBackgroundColor(ContextCompat.getColorStateList(activity, R.color.chip_green_color));
                    }
                }

            }
            viewHolder.tvOrderTypeForDelay.setVisibility(View.GONE);
            viewHolder.tvCustomerName.setText("#" + order.order_number);
            if (isDinein) {
                viewHolder.tvOrderStatus.setText("Dine in - " + paymentStatus);
                addressString = "Table No: " + order.table_no;
                timeString = "Guests: " + order.no_of_guest;
            } else {
                String orderType = (order.order_type).substring(0, 1).toUpperCase() + (order.order_type).substring(1);
                if (isPreOrder) {
                    viewHolder.tvOrderStatus.setText("Pre-order: " + CommonFunctions.formatUnknownDateTime(order.delivery_date, MyApp.DEL_TIMEFORMAT, "dd/MM/yyyy"));
                    viewHolder.tvCustomerName.setText(orderType + " - " + paymentStatus + "\n" + "#" + order.order_number);
                } else {
                    if (delayed) {
                        viewHolder.tvOrderStatus.setText("Delayed");
                        viewHolder.tvOrderTypeForDelay.setVisibility(View.VISIBLE);
                        viewHolder.tvOrderTypeForDelay.setText(orderType + " - " + paymentStatus);
                        viewHolder.tvCustomerName.setText("#" + order.order_number);
                    } else {
                        viewHolder.tvOrderTypeForDelay.setVisibility(View.GONE);
                        viewHolder.tvOrderStatus.setText(orderType + " - " + paymentStatus);
                    }
                }
                addressString = order.address;
                timeString = CommonFunctions.convertTimeFormat(order.delivery_time);
            }

            viewHolder.tvAddress.setText(addressString);
            viewHolder.tvTime.setText(timeString);
            viewHolder.chip.setText(chipMessage);
            if (Validators.isNullOrEmpty(addressString) || (order.order_type != null && order.order_type.equalsIgnoreCase("pickup"))) {
                viewHolder.tvAddress.setVisibility(View.GONE);
            } else {
                viewHolder.tvAddress.setVisibility(View.VISIBLE);
            }
            if (Validators.isNullOrEmpty(timeString)) {
                viewHolder.tvTime.setVisibility(View.GONE);
            } else {
                viewHolder.tvTime.setVisibility(View.VISIBLE);
            }

            viewHolder.tvView.setOnClickListener(v -> {
                if (viewClickListener != null) {
                    viewClickListener.onItemClick(v, position, order);
                }
            });
            viewHolder.ivPrint.setOnClickListener(v -> {
                if (printClickListener != null) {
                    printClickListener.onItemClick(v, position, order);
                }
            });
            viewHolder.chip.setOnClickListener(v -> {
                if (itemViewClickListener != null) {
                    itemViewClickListener.onItemClick(v, position, order);
                }
            });
            viewHolder.cvOrder.setOnClickListener(v -> {
                if (viewClickListener != null) {
                    viewClickListener.onItemClick(v, position, order);
                }
            });
            viewHolder.chipRtc.setOnClickListener(v -> {
                if (rtcClicked != null) {
                    rtcClicked.onItemClick(v, position, order);
                }
            });

            viewHolder.tvOrderId.setVisibility(View.GONE);
        }
    }

    @Override
    public int getItemCount() {
        return ordersList.size();
    }

    @Override
    public int getHeaderPositionForItem(int itemPosition) {
        int headerPosition = 0;
        do {
            if (this.isHeader(itemPosition)) {
                headerPosition = itemPosition;
                break;
            }
            itemPosition -= 1;
        } while (itemPosition >= 0);
        return headerPosition;
    }

    @Override
    public int getHeaderLayout(int headerPosition) {
        return R.layout.item_orders_status_indicator;
    }

    @Override
    public void bindHeaderData(View header, int headerPosition) {
        TextView tvTitle = header.findViewById(R.id.tvTitle);
        OrderStatusIndicatorHeader indicatorHeader = (OrderStatusIndicatorHeader) ordersList.get(headerPosition);
        tvTitle.setText(indicatorHeader.title);
    }

    @Override
    public boolean isHeader(int itemPosition) {
        return getItemViewType(itemPosition) == R.layout.item_orders_status_indicator;
    }

    protected static class OrderViewHolder extends RecyclerView.ViewHolder {
        private MaterialCardView cvOrder;
        private TextView tvOrderStatus, tvCustomerName, tvAddress, tvTime, tvView, tvOrderId, tvOrderTypeForDelay;
        private AppCompatImageView ivPrint;
        private Chip chip, chipRtc;

        public OrderViewHolder(@NonNull View itemView) {
            super(itemView);
            cvOrder = itemView.findViewById(R.id.cvOrder);
            tvOrderStatus = itemView.findViewById(R.id.tvOrderStatus);
            tvCustomerName = itemView.findViewById(R.id.tvName);
            tvAddress = itemView.findViewById(R.id.tvAddress);
            tvTime = itemView.findViewById(R.id.tvTime);
            ivPrint = itemView.findViewById(R.id.ivPrint);
            chip = itemView.findViewById(R.id.Chip);
            chipRtc = itemView.findViewById(R.id.chipRtc);
            tvView = itemView.findViewById(R.id.tvView);
            tvOrderId = itemView.findViewById(R.id.tvOrderId);
            tvOrderTypeForDelay = itemView.findViewById(R.id.tvOrderTypeForDelay);
        }
    }

    protected static class OrderStatusHeaderViewHolder extends RecyclerView.ViewHolder {
        private TextView tvTitle;

        public OrderStatusHeaderViewHolder(@NonNull View itemView) {
            super(itemView);
            tvTitle = itemView.findViewById(R.id.tvTitle);
        }
    }
}
