package com.tiffintom.partner1.fragments;

import android.bluetooth.BluetoothDevice;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CompoundButton;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.widget.SwitchCompat;
import androidx.coordinatorlayout.widget.CoordinatorLayout;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.common.Priority;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.JSONObjectRequestListener;
import com.androidnetworking.interfaces.ParsedRequestListener;
import com.google.android.material.snackbar.Snackbar;
import com.google.gson.Gson;
import com.tiffintom.partner1.MyApp;
import com.tiffintom.partner1.R;
import com.tiffintom.partner1.adapters.OrdersAdapter;
import com.tiffintom.partner1.base.BaseFragment;
import com.tiffintom.partner1.common.CommonFunctions;
import com.tiffintom.partner1.common.StickHeaderItemDecoration;
import com.tiffintom.partner1.common.Validators;
import com.tiffintom.partner1.common.bluetoothprinter.BluetoothPrinter;
import com.tiffintom.partner1.common.printer.CS20PrintHelper;
import com.tiffintom.partner1.common.printer.SunmiPrinter;
import com.tiffintom.partner1.common.printer.ZoneRichPrinter;
import com.tiffintom.partner1.interfaces.DialogDismissListener;
import com.tiffintom.partner1.interfaces.FilterDialogDismissListener;
import com.tiffintom.partner1.models.OrderDetail;
import com.tiffintom.partner1.models.OrderStatusIndicatorHeader;
import com.tiffintom.partner1.models.RestaurantOrder;
import com.tiffintom.partner1.models.RestaurantOrderResponse;
import com.tiffintom.partner1.network.ApiEndPoints;
import com.tiffintom.partner1.utils.Constants;
import com.tiffintom.partner1.utils.LogUtils;
import com.tiffintom.partner1.utils.ToastUtils;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;

/**
 * Created by Amrish on 09-11-2020.
 */
public class OrdersAllFragment extends BaseFragment {
    private RestaurantOrderResponse ordersResponse;
    private ArrayList<Object> ordersList = new ArrayList<>();
    private RecyclerView rvOrders;
    private OrdersAdapter ordersAdapter;
    private SwitchCompat switchCompat;
    private LinearLayout llLoading, llData, llNoRecord;
    private boolean custom = false;
    private String sort_by = "Today";
    private String toDate, fromDate;
    private final Calendar fromCalendar = Calendar.getInstance();
    private final Calendar toCalendar = Calendar.getInstance();
    private LinearLayout llFilter;
    private TextView tvFilter;
    private SunmiPrinter sunmiPrinter;
    private ZoneRichPrinter zoneRichPrinter;
    private CS20PrintHelper cs20PrintHelper;
    private boolean justPrint = false;
    private ImageView ivBack;
    private Snackbar greySnackBar;
    private CoordinatorLayout coordinatorLayout;
    private BluetoothPrinter bluetoothPrinter;

    public static OrdersAllFragment getInstance() {
        return new OrdersAllFragment();
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_orders_all, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        try {
            updateOrderTakingStatus();
            setListeners();
            fetchData();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void manageIntents() {
        super.manageIntents();
    }

    RestaurantOrder clickedOrder;

    @Override
    protected void initViews(View view) {
        super.initViews(view);
        try {
            coordinatorLayout = view.findViewById(R.id.coordinatorLayout);
            zoneRichPrinter = new ZoneRichPrinter(getActivity());
            if (myApp.getMyPreferences().getDefaultPrinter().equalsIgnoreCase(getString(R.string.ubsidi_cs30))) {
                cs20PrintHelper = new CS20PrintHelper();
            }
            sunmiPrinter = new SunmiPrinter(getActivity());
            bluetoothPrinter = new BluetoothPrinter(getActivity());
            llFilter = view.findViewById(R.id.llFiler);
            tvFilter = view.findViewById(R.id.tvFilter);
            rvOrders = view.findViewById(R.id.rvOrders);
            switchCompat = view.findViewById(R.id.switchOrders);
            llLoading = view.findViewById(R.id.llLoading);
            llData = view.findViewById(R.id.llData);
            llNoRecord = view.findViewById(R.id.llNoRecord);
            ivBack = view.findViewById(R.id.ivBack);

            ordersAdapter = new OrdersAdapter(getActivity(), ordersList, (view1, position, data) -> {
                RestaurantOrder order = (RestaurantOrder) data;
                int orderType = -1;
                if (Validators.isNullOrEmpty(order.order_type)) {
                    orderType = 0;
                } else if (order.order_type.equalsIgnoreCase("delivery")) {
                    orderType = 1;
                } else if (order.order_type.equalsIgnoreCase("pickup")) {
                    orderType = 2;
                }
                if (order.status.equalsIgnoreCase("pending")) {
                    OrderViewBottomSheetFragment bottomSheetFragment = OrderViewBottomSheetFragment.getInstance(orderType, order.id);
                    bottomSheetFragment.show(getChildFragmentManager(), "orders");
                    bottomSheetFragment.setDialogDismissListener(o -> {
                        if (o instanceof String) {
                            if (((String) o).equalsIgnoreCase("refresh")) {
                                fetchData();
                            }
                        }
                    });

                } else if (order.status.equalsIgnoreCase("accepted")) {
                    openConfirmationDialog();

                }
            }, (view1, position, data) -> {
                RestaurantOrder order = (RestaurantOrder) data;
                int orderType = -1;
                if (Validators.isNullOrEmpty(order.order_type)) {
                    orderType = 0;
                } else if (order.order_type.equalsIgnoreCase("delivery")) {
                    orderType = 1;
                } else if (order.order_type.equalsIgnoreCase("pickup")) {
                    orderType = 2;
                }
                OrderViewBottomSheetFragment bottomSheetFragment = OrderViewBottomSheetFragment.getInstance(orderType, order.id);
                bottomSheetFragment.show(getChildFragmentManager(), "orders");
                bottomSheetFragment.setDialogDismissListener(o -> {
                    if (o instanceof String) {
                        if (((String) o).equalsIgnoreCase("refresh")) {
                            fetchData();
                        }
                    }
                });
            }, (view1, position, data) -> {
                // Print clicked
                justPrint = true;
                fetchOrderDetail((RestaurantOrder) data);
            }, (view1, position, data) -> {
                //RTC clicked
            });
            rvOrders.setLayoutManager(new LinearLayoutManager(getActivity(), RecyclerView.VERTICAL, false));
            rvOrders.setNestedScrollingEnabled(false);
            rvOrders.setAdapter(ordersAdapter);
            rvOrders.addItemDecoration(new StickHeaderItemDecoration(ordersAdapter));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void setListeners() {

        try {
            ivBack.setOnClickListener(v -> getActivity().onBackPressed());
            switchCompat.setOnCheckedChangeListener(checkedChangeListener);
            llFilter.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    FilterSelectionFragment filterSelectionFragment = FilterSelectionFragment.getInstance(fromDate, toDate);
                    filterSelectionFragment.show(getChildFragmentManager(), "filters");
                    filterSelectionFragment.setFilterDialogDismissListener(new FilterDialogDismissListener() {
                        @Override
                        public void onDialogDismiss(Date from, Date to) {

                        }

                        @Override
                        public void onDialogDismiss(String from, String to) {

                        }

                        @Override
                        public void onDialogDismiss(String filter, String from, String to) {
                            if (filter.equalsIgnoreCase("custom")) {
                                custom = true;
                                fromDate = from;
                                toDate = to;
                                tvFilter.setText(fromDate + " - " + toDate);
                            } else {
                                custom = false;
                                sort_by = filter.replace(" ", "");
                                tvFilter.setText(filter);
                            }
                            fetchData();
                        }
                    });
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void fetchData() {
        new Thread(this::fetchOrders).start();
    }

    private void fetchOrders() {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                    llLoading.setVisibility(View.VISIBLE);
                });
            }
            //'ThisWeek','ThisMonth','ThisYear','Today', 'Yesterday'
            HashMap<String, String> params = new HashMap<>();
            params.put("restaurant_id", myApp.getMyPreferences().getLoggedInRestaurant().id);
            if (!custom) {
                //params.put("sortby", sort_by);
                if (sort_by.equalsIgnoreCase("Today")) {
                    params.put("from_date", toDate.replace("/", "-"));
                    params.put("to_date", toDate.replace("/", "-"));
                } else if (sort_by.equalsIgnoreCase("ThisWeek")) {
                    params.put("to_date", CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd").replace("/", "-"));
                    params.put("from_date", CommonFunctions.getPreviousDate(CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd").replace("/", "-"), -6));
                } else if (sort_by.equalsIgnoreCase("ThisMonth")) {
                    params.put("to_date", CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd").replace("/", "-"));
                    params.put("from_date", CommonFunctions.getPreviousDate(CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd").replace("/", "-"), -30));
                } else if (sort_by.equalsIgnoreCase("ThisYear")) {
                    params.put("to_date", CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd").replace("/", "-"));
                    params.put("from_date", CommonFunctions.getPreviousDate(CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd").replace("/", "-"), -365));
                } else {
                    params.put("from_date", fromDate.replace("/", "-"));
                    params.put("to_date", toDate.replace("/", "-"));
                }
            } else {
                params.put("from_date", fromDate.replace("/", "-"));
                params.put("to_date", toDate.replace("/", "-"));
            }

            AndroidNetworking.get(ApiEndPoints.orders + "all-orders")
                    // .addQueryParameter("action", "StoreorderList")
                    .addQueryParameter(params)
                    .build()
                    .getAsObject(RestaurantOrderResponse.class, new ParsedRequestListener<RestaurantOrderResponse>() {
                        @Override
                        public void onResponse(RestaurantOrderResponse response) {
                            try {
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        llLoading.setVisibility(View.GONE);
                                        llData.setVisibility(View.VISIBLE);
                                        llNoRecord.setVisibility(View.GONE);
                                    });
                                }
                                if (response.normal_orders != null && response.normal_orders.size() > 0) {
                                    ordersResponse = response;
                                    setupAdapter();
                                    ordersAdapter.notifyDataSetChanged();
                                } else {
                                    getActivity().runOnUiThread(() -> {
                                        llNoRecord.setVisibility(View.VISIBLE);
                                    });
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        llLoading.setVisibility(View.GONE);
                                        llNoRecord.setVisibility(View.VISIBLE);
                                        ordersList.clear();
                                        ordersAdapter.notifyDataSetChanged();
                                    });
                                }
                                anError.printStackTrace();
                                if (!CommonFunctions.isConnected(getActivity())) {
                                    myApp.noInternet(getActivity());
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    private void openConfirmationDialog() {
        try {
            ConfirmationDialogFragment confirmationDialogFragment = ConfirmationDialogFragment.getInstance("Accept Booking", "Do you want to confirm this booking?", false);
            confirmationDialogFragment.show(getChildFragmentManager(), "order_accepted");
            confirmationDialogFragment.setDialogDismissListener(o -> {
                String action = (String) o; //confirm, cancel
                if (action.equalsIgnoreCase("confirm")) {
                    changeOrderStatus(clickedOrder.id);
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void changeOrderStatus(String order_id) {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> progressDialog.show());
            }
            HashMap<String, String> params = new HashMap<>();
            params.put("status", "Delivered");
            AndroidNetworking.post(ApiEndPoints.orders + order_id + "/change-status")
                    .addQueryParameter(params)
                    .build()
                    .getAsJSONObject(new JSONObjectRequestListener() {
                        @Override
                        public void onResponse(JSONObject response) {
                            try {
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> progressDialog.dismiss());
                                }
                                fetchData();
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                anError.printStackTrace();
                                LogUtils.e("Order Status Change error");
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> progressDialog.dismiss());
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    CompoundButton.OnCheckedChangeListener checkedChangeListener = new CompoundButton.OnCheckedChangeListener() {
        @Override
        public void onCheckedChanged(CompoundButton compoundButton, boolean b) {
            try {
                EnterPasswordDialogFragment enterPasswordDialogFragment = EnterPasswordDialogFragment.getInstance("online_order", b ? "Yes" : "No");
                enterPasswordDialogFragment.show(getChildFragmentManager(), "enter_password");
                enterPasswordDialogFragment.setCancelable(false);
                enterPasswordDialogFragment.setDialogDismissListener(changeOrderAcceptingStatusDialogListener);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    };
    DialogDismissListener changeOrderAcceptingStatusDialogListener = new DialogDismissListener() {
        @Override
        public void onDialogDismiss(Object o) {
            try {
                switchCompat.setOnCheckedChangeListener(null);
                updateOrderTakingStatus();
                switchCompat.setOnCheckedChangeListener(checkedChangeListener);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    };

    private void updateOrderTakingStatus() {
        try {
            switchCompat.setChecked(myApp.getMyPreferences().getLoggedInRestaurant().online_order.equalsIgnoreCase("Yes"));
            fromCalendar.add(Calendar.DAY_OF_MONTH, -1);

            fromDate = CommonFunctions.formatMiliToDesireFormat(fromCalendar.getTimeInMillis(), "yyyy-MM-dd");
            toDate = CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void setupAdapter() {
        try {
            if (ordersResponse != null) {
                ordersList.clear();
                int completedOrders = 0;
                if (ordersResponse.normal_orders != null) {
                    completedOrders = completedOrders + ordersResponse.normal_orders.size();
                }
                if (ordersResponse.dinein_orders != null) {
                    completedOrders = completedOrders + ordersResponse.dinein_orders.size();
                }
                if (completedOrders > 0) {
                    ArrayList<RestaurantOrder> orders = new ArrayList<>();
                    ordersList.add(new OrderStatusIndicatorHeader("Completed order (" + completedOrders + ")"));
                    if (ordersResponse.normal_orders != null)
                        orders.addAll(ordersResponse.normal_orders);
                    if (ordersResponse.dinein_orders != null)
                        orders.addAll(ordersResponse.dinein_orders);
                    ordersList.addAll(sortOrdersByNew(orders));
                }
            } else {
                ordersList.clear();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private ArrayList<RestaurantOrder> sortOrdersByNew(ArrayList<RestaurantOrder> orders) {
        Collections.sort(orders, (restaurantOrder, t1) -> {
            Date orderDateLHS;
            if (!Validators.isNullOrEmpty(restaurantOrder.order_type)) {
                String date = restaurantOrder.delivery_date.split("T")[0] + " " + restaurantOrder.delivery_time;
                orderDateLHS = CommonFunctions.convertStringDateToDate(date, "yyyy-MM-dd hh:mm a");
            } else {
                orderDateLHS = CommonFunctions.convertStringDateToDate(CommonFunctions.formatUnknownDateTime(restaurantOrder.created, MyApp.PHP_DATE_TIME_FORMAT_ZULU, "dd-MM-yyyy hh:mm a"), "dd-MM-yyyy hh:mm a");
            }

            Date orderDateRHS;
            if (!Validators.isNullOrEmpty(t1.order_type)) {
                String date = t1.delivery_date.split("T")[0] + " " + t1.delivery_time;
                orderDateRHS = CommonFunctions.convertStringDateToDate(date, "yyyy-MM-dd hh:mm a");
            } else {
                orderDateRHS = CommonFunctions.convertStringDateToDate(CommonFunctions.formatUnknownDateTime(t1.created, MyApp.PHP_DATE_TIME_FORMAT_ZULU, "dd-MM-yyyy hh:mm a"), "dd-MM-yyyy hh:mm a");
            }

            return orderDateLHS.compareTo(orderDateRHS);
        });
        return orders;
    }

    private void fetchOrderDetail(RestaurantOrder order) {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                    progressDialog.show();
                });
            }
            AndroidNetworking.get(ApiEndPoints.orders + order.id).setPriority(Priority.IMMEDIATE)
                    .build()
                    .getAsJSONObject(new JSONObjectRequestListener() {
                        @Override
                        public void onResponse(JSONObject response) {
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> {
                                    progressDialog.dismiss();
                                });
                            }
                            try {
                                OrderDetail orderDetail = new Gson().fromJson(response.toString(), OrderDetail.class);
                                createPrint(orderDetail);
                            } catch (Exception e) {
                                e.printStackTrace();
                                CommonFunctions.showSnackBar(getActivity(), coordinatorLayout, "Order not found");
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                anError.printStackTrace();
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        progressDialog.dismiss();
                                    });
                                }
                                LogUtils.e("ERROR IN ORDER VIEW");
                                LogUtils.e(anError.getErrorBody());
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void createPrint(OrderDetail orderDetail) {
        try {
            if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains("imin")) {
                printIMin(orderDetail, false);
            } else if (myApp.getMyPreferences().getDefaultPrinter().equalsIgnoreCase(getString(R.string.ubsidi_cs30))) {
                printCS20(orderDetail, false);
            } else if (!myApp.getMyPreferences().getDefaultPrinter().equalsIgnoreCase("mobile")) {

                if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains("ip printer")) {
                    if (zoneRichPrinter == null) {
                        CommonFunctions.showSnackBar(getActivity(), coordinatorLayout, "Printer is null");
                    } else {
                        printZonerich(orderDetail, false);
                    }
                } else if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains(Constants.KP_80) || myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains(Constants.KP80)) {
                    try {
                        if (bluetoothPrinter != null) {
                            BluetoothDevice device = bluetoothPrinter.getConnectedPrinter();
                            if (device != null) {
                                CommonFunctions.functionThatDelay(200);
                                printBT(orderDetail, false);
                            } else {
                                ToastUtils.makeToast(getActivity(), "No bluetooth device found.");
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else {
                    printSunmi(orderDetail, false);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printIMin(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            myApp.iMinPrinterUtils.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    printIMin(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printZonerich(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            zoneRichPrinter.connectPrinter(myApp.getMyPreferences().getPrinterIP());
            zoneRichPrinter.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    zoneRichPrinter.connectPrinter(myApp.getMyPreferences().getPrinterIP());
                    printZonerich(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printCS20(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            cs20PrintHelper.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    printCS20(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printSunmi(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            sunmiPrinter.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    printSunmi(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printBT(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            bluetoothPrinter.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    printBT(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onStop() {
        try {
            if (getActivity() != null && progressDialog != null) {
                progressDialog.dismiss();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        super.onStop();
    }

}
