package com.tiffintom.partner1.fragments;

import android.bluetooth.BluetoothDevice;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CompoundButton;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.widget.SwitchCompat;
import androidx.coordinatorlayout.widget.CoordinatorLayout;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentTransaction;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.common.Priority;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.JSONObjectRequestListener;
import com.androidnetworking.interfaces.ParsedRequestListener;
import com.google.android.material.snackbar.Snackbar;
import com.google.gson.Gson;
import com.tiffintom.partner1.MyApp;
import com.tiffintom.partner1.R;
import com.tiffintom.partner1.activities.SplashActivity;
import com.tiffintom.partner1.adapters.OrdersAdapter;
import com.tiffintom.partner1.base.BaseFragment;
import com.tiffintom.partner1.common.CommonFunctions;
import com.tiffintom.partner1.common.StickHeaderItemDecoration;
import com.tiffintom.partner1.common.Validators;
import com.tiffintom.partner1.common.bluetoothprinter.BluetoothPrinter;
import com.tiffintom.partner1.common.printer.CS20PrintHelper;
import com.tiffintom.partner1.common.printer.SunmiPrinter;
import com.tiffintom.partner1.common.printer.ZoneRichPrinter;
import com.tiffintom.partner1.interfaces.DialogDismissListener;
import com.tiffintom.partner1.models.OrderDetail;
import com.tiffintom.partner1.models.OrderStatusIndicatorHeader;
import com.tiffintom.partner1.models.RestaurantOrder;
import com.tiffintom.partner1.models.RestaurantOrderResponse;
import com.tiffintom.partner1.network.ApiEndPoints;
import com.tiffintom.partner1.utils.Constants;
import com.tiffintom.partner1.utils.LogUtils;
import com.tiffintom.partner1.utils.ToastUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Objects;

/**
 * Created by Amrish on 09-11-2020.
 */
public class OrdersFragment extends BaseFragment {
    private RestaurantOrderResponse ordersResponse;
    private ArrayList<Object> ordersList = new ArrayList<>();
    private RecyclerView rvOrders;
    private OrdersAdapter ordersAdapter;
    private SwitchCompat switchCompat;
    private LinearLayout llLoading, llData, llNoRecord;
    private TextView tvOrders;
    private SwipeRefreshLayout swipeRefreshLayout;
    private SunmiPrinter sunmiPrinter;
    private BroadcastReceiver broadcastReceiver, orderStatusChangedReceiver;
    private ZoneRichPrinter zoneRichPrinter;
    private Snackbar snackbar, greySnackBar;
    private CoordinatorLayout coordinatorLayout;
    private BluetoothPrinter bluetoothPrinter;
    private CS20PrintHelper cs20PrintHelper;

    public static OrdersFragment getInstance() {
        return new OrdersFragment();
    }

    private boolean isRequestOnGoing = false, justPrint = false;

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_orders, container, false);
    }

    @Override
    protected void initViews(View view) {
        super.initViews(view);
        try {
            sunmiPrinter = new SunmiPrinter(getActivity());
            zoneRichPrinter = new ZoneRichPrinter(getActivity());
            bluetoothPrinter = new BluetoothPrinter(getActivity());
            if (myApp.getMyPreferences().getDefaultPrinter().equalsIgnoreCase(getString(R.string.ubsidi_cs30))) {
                cs20PrintHelper = new CS20PrintHelper();
            }
            coordinatorLayout = view.findViewById(R.id.coordinatorLayout);
            rvOrders = view.findViewById(R.id.rvOrders);
            switchCompat = view.findViewById(R.id.switchOrders);
            llLoading = view.findViewById(R.id.llLoading);
            tvOrders = view.findViewById(R.id.tvAllOrders);
            llData = view.findViewById(R.id.llRootData);
            llNoRecord = view.findViewById(R.id.llNoRecord);
            swipeRefreshLayout = view.findViewById(R.id.swipeRefreshLayout);

            ordersAdapter = new OrdersAdapter(getActivity(), ordersList, (view1, position, data) -> {
                RestaurantOrder order = (RestaurantOrder) data;
                int orderType = -1;
                if (Validators.isNullOrEmpty(order.order_type)) {
                    orderType = 0;
                } else if (order.order_type.equalsIgnoreCase("delivery")) {
                    orderType = 1;
                } else if (order.order_type.equalsIgnoreCase("pickup")) {
                    orderType = 2;
                }
                if (order.status.equalsIgnoreCase("pending")) {
                    OrderViewBottomSheetFragment bottomSheetFragment = OrderViewBottomSheetFragment.getInstance(orderType, order.id);
                    bottomSheetFragment.show(getChildFragmentManager(), "orders");
                    bottomSheetFragment.setDialogDismissListener(o -> {
                        if (o instanceof String) {
                            if (((String) o).equalsIgnoreCase("refresh")) {
                                fetchData();
                            }
                        }
                    });

                } else if (order.status.equalsIgnoreCase("accepted")) {
                    String orderTitle = "", orderDescription = "";
                    if (Validators.isNullOrEmpty(order.order_type)) {
                        orderTitle = "Ready to serve";
                        orderDescription = "Do you want to confirm mark this order as ready to serve?";
                    } else {
                        if (order.order_type.equalsIgnoreCase("delivery")) {
                            orderTitle = "Mark as delivered";
                            orderDescription = "Do you want to mark this order as delivered?";
                        } else if (order.order_type.equalsIgnoreCase("pickup")) {
                            orderTitle = "Mark as picked up";
                            orderDescription = "Do you want to mark this order as picked up?";
                        }
                    }
                    openConfirmationDialog(order, orderTitle, orderDescription);

                }
            }, (view1, position, data) -> {
                RestaurantOrder order = (RestaurantOrder) data;
                int orderType = -1;
                if (Validators.isNullOrEmpty(order.order_type)) {
                    orderType = 0;
                } else if (order.order_type.equalsIgnoreCase("delivery")) {
                    orderType = 1;
                } else if (order.order_type.equalsIgnoreCase("pickup")) {
                    orderType = 2;
                }
                OrderViewBottomSheetFragment bottomSheetFragment = OrderViewBottomSheetFragment.getInstance(orderType, order.id);
                bottomSheetFragment.show(getChildFragmentManager(), "orders");
                bottomSheetFragment.setDialogDismissListener(o -> {
                    if (o instanceof String) {
                        if (((String) o).equalsIgnoreCase("refresh")) {
                            fetchData();
                        }
                    }
                });
            }, (view1, position, data) -> {
                // Print clicked
                justPrint = true;
                fetchOrderDetail((RestaurantOrder) data);
            }, (view1, position, data) -> {
                // RTC Clicked
                sendRTC(((RestaurantOrder) data).id);
            });

            rvOrders.setLayoutManager(new LinearLayoutManager(getActivity(), RecyclerView.VERTICAL, false));
            rvOrders.setNestedScrollingEnabled(false);
            rvOrders.setAdapter(ordersAdapter);
            rvOrders.addItemDecoration(new StickHeaderItemDecoration(ordersAdapter));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        try {
            updateOrderTakingStatus();
            setListeners();
            fetchData();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void setListeners() {
        try {
            switchCompat.setOnCheckedChangeListener(checkedChangeListener);
            swipeRefreshLayout.setOnRefreshListener(this::fetchData);
            tvOrders.setOnClickListener(view -> {
                FragmentTransaction ft = getFragmentManager().beginTransaction();
                ft.replace(R.id.nav_host_fragment, OrdersAllFragment.getInstance(), "all_orders");
                ft.addToBackStack("let_go_back");
                ft.commit();
            });
            broadcastReceiver = new BroadcastReceiver() {
                public void onReceive(Context context, Intent intent) {

                    boolean refresh = intent.getBooleanExtra("refresh", false);
                    boolean forOrders = intent.getBooleanExtra("orders", false);
                    if (refresh && forOrders && !isRequestOnGoing) {
                        swipeRefreshLayout.setRefreshing(true);
                        fetchData();
                    }
                }
            };
            orderStatusChangedReceiver = new BroadcastReceiver() {
                @Override
                public void onReceive(Context context, Intent intent) {
                    String message = intent.getStringExtra("message");
                    boolean reject = intent.getBooleanExtra("rejected", false);
                    String orderDetails = intent.getStringExtra("orderdetail");

                    OrderDetail orderDetail = new Gson().fromJson(orderDetails, OrderDetail.class);
                    int orderType = -1;
                    if (Validators.isNullOrEmpty(orderDetail.order_type)) {
                        orderType = 0;
                    } else if (orderDetail.order_type.equalsIgnoreCase("delivery")) {
                        orderType = 1;
                    } else if (orderDetail.order_type.equalsIgnoreCase("pickup")) {
                        orderType = 2;
                    }
                    try {
                        int finalOrderType = orderType;
                        snackbar = Snackbar.make(coordinatorLayout, message, Snackbar.LENGTH_LONG)
                                .setActionTextColor(getResources().getColor(R.color.white))
                                .setAction("view", v -> {
                                    OrderViewBottomSheetFragment bottomSheetFragment = OrderViewBottomSheetFragment.getInstance(finalOrderType, orderDetail.id);
                                    bottomSheetFragment.show(getActivity().getSupportFragmentManager(), "order");
                                    bottomSheetFragment.setDialogDismissListener(o -> {
                                        if (o instanceof String) {
                                            if (((String) o).equalsIgnoreCase("refresh")) {
                                                fetchData();
                                            }
                                        }
                                    });
                                });
                        if (reject) {
                            snackbar.setBackgroundTint(ContextCompat.getColor(getActivity(), R.color.orders_card_bg_orange));
                        } else {
                            snackbar.setBackgroundTint(ContextCompat.getColor(getActivity(), R.color.orders_card_bg_green));
                        }
                        snackbar.show();
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            };
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    CompoundButton.OnCheckedChangeListener checkedChangeListener = new CompoundButton.OnCheckedChangeListener() {
        @Override
        public void onCheckedChanged(CompoundButton compoundButton, boolean b) {
            try {
                EnterPasswordDialogFragment enterPasswordDialogFragment = EnterPasswordDialogFragment.getInstance("online_order", b ? "Yes" : "No");
                enterPasswordDialogFragment.show(getChildFragmentManager(), "enter_password");
                enterPasswordDialogFragment.setCancelable(false);
                enterPasswordDialogFragment.setDialogDismissListener(changeOrderAcceptingStatusDialogListener);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    };

    DialogDismissListener changeOrderAcceptingStatusDialogListener = new DialogDismissListener() {
        @Override
        public void onDialogDismiss(Object o) {
            try {
                switchCompat.setOnCheckedChangeListener(null);
                updateOrderTakingStatus();
                switchCompat.setOnCheckedChangeListener(checkedChangeListener);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    };

    private void updateOrderTakingStatus() {
        switchCompat.setChecked(myApp.getMyPreferences().getLoggedInRestaurant().online_order.equalsIgnoreCase("Yes"));
    }

    private void setupAdapter() {
        try {
            if (ordersResponse != null) {
                ordersList.clear();
                int pendingOrdersCount = 0;
                if (ordersResponse.pending_orders != null) {
                    pendingOrdersCount = pendingOrdersCount + ordersResponse.pending_orders.size();
                }
                if (ordersResponse.pending_dinein_orders != null) {
                    pendingOrdersCount = pendingOrdersCount + ordersResponse.pending_dinein_orders.size();
                }

                if (pendingOrdersCount > 0) {
                    // Check Pre-Order
                    ArrayList<RestaurantOrder> orders = new ArrayList<>();
                    if (ordersResponse.pending_orders != null)
                        orders.addAll(ordersResponse.pending_orders);
                    if (ordersResponse.pending_dinein_orders != null)
                        orders.addAll(ordersResponse.pending_dinein_orders);

                    ArrayList<RestaurantOrder> pendingOrders = new ArrayList<>();
                    ArrayList<RestaurantOrder> preOrders = new ArrayList<>();

                    for (RestaurantOrder order : orders) {
                        if (!Validators.isNullOrEmpty(order.order_type)) {
                            // Check pre-order
                            if (CommonFunctions.isPreOrderWithoutStatus(order.delivery_date)) {
                                pendingOrders.add(order);
                            } else {
                                preOrders.add(order);
                            }
                        } else {
                            pendingOrders.add(order);
                        }
                    }
                    if (preOrders.size() > 0) {
                        ordersList.add(new OrderStatusIndicatorHeader("Pre order (" + preOrders.size() + ")"));
                        ordersList.addAll(sortOrdersByNew(preOrders));
                    }
                    if (pendingOrders.size() > 0) {
                        ordersList.add(new OrderStatusIndicatorHeader("New order (" + pendingOrders.size() + ")"));
                        ordersList.addAll(sortOrdersByNew(pendingOrders));
                    }
                }

                int inProgressOrders = 0;
                if (ordersResponse.accepted_orders != null) {
                    inProgressOrders = inProgressOrders + ordersResponse.accepted_orders.size();
                }
                if (ordersResponse.accepted_dinein_orders != null) {
                    inProgressOrders = inProgressOrders + ordersResponse.accepted_dinein_orders.size();
                }
                if (inProgressOrders > 0) {

                    ArrayList<RestaurantOrder> orders = new ArrayList<>();
                    if (ordersResponse.accepted_orders != null)
                        orders.addAll(ordersResponse.accepted_orders);
                    if (ordersResponse.accepted_dinein_orders != null)
                        orders.addAll(ordersResponse.accepted_dinein_orders);

                    ordersList.add(new OrderStatusIndicatorHeader("Order in progress (" + inProgressOrders + ")"));
                    ordersList.addAll(sortOrdersByNew(orders));
                }

                int completedOrders = 0;
                if (ordersResponse.completed_orders != null) {
                    completedOrders = completedOrders + ordersResponse.completed_orders.size();
                }
                if (ordersResponse.completed_dinein_orders != null) {
                    completedOrders = completedOrders + ordersResponse.completed_dinein_orders.size();
                }
                if (completedOrders > 0) {
                    ArrayList<RestaurantOrder> orders = new ArrayList<>();
                    ordersList.add(new OrderStatusIndicatorHeader("Completed order (" + completedOrders + ")"));
                    if (ordersResponse.completed_orders != null)
                        orders.addAll(ordersResponse.completed_orders);
                    if (ordersResponse.completed_dinein_orders != null)
                        orders.addAll(ordersResponse.completed_dinein_orders);
                    ordersList.addAll(sortOrdersByNew(orders));
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private ArrayList<RestaurantOrder> sortOrdersByNew(ArrayList<RestaurantOrder> orders) {
        Collections.sort(orders, (restaurantOrder, t1) -> {
            Date orderDateLHS;
            if (!Validators.isNullOrEmpty(restaurantOrder.order_type)) {
                String date = restaurantOrder.delivery_date.split("T")[0] + " " + CommonFunctions.convertTimeFormat(restaurantOrder.delivery_time);
                orderDateLHS = CommonFunctions.convertStringDateToDate(date, "yyyy-MM-dd hh:mm a");
            } else {
                orderDateLHS = CommonFunctions.convertStringDateToDate(CommonFunctions.formatUnknownDateTime(restaurantOrder.created, MyApp.PHP_DATE_TIME_FORMAT_ZULU, "dd-MM-yyyy hh:mm a"), "dd-MM-yyyy hh:mm a");
            }

            Date orderDateRHS;
            if (!Validators.isNullOrEmpty(t1.order_type)) {
                String date = t1.delivery_date.split("T")[0] + " " + CommonFunctions.convertTimeFormat(t1.delivery_time);
                orderDateRHS = CommonFunctions.convertStringDateToDate(date, "yyyy-MM-dd hh:mm a");
            } else {
                orderDateRHS = CommonFunctions.convertStringDateToDate(CommonFunctions.formatUnknownDateTime(t1.created, MyApp.PHP_DATE_TIME_FORMAT_ZULU, "dd-MM-yyyy hh:mm a"), "dd-MM-yyyy hh:mm a");
            }

            return orderDateLHS.compareTo(orderDateRHS);
        });
        return orders;
    }

    private void openConfirmationDialog(RestaurantOrder order, String orderTitle, String orderDescription) {
        try {
            ConfirmationDialogFragment confirmationDialogFragment = ConfirmationDialogFragment.getInstance(orderTitle, orderDescription, false);
            confirmationDialogFragment.show(getChildFragmentManager(), "order_accepted");
            confirmationDialogFragment.setDialogDismissListener(o -> {
                String action = (String) o; //confirm, cancel
                if (action.equalsIgnoreCase("confirm")) {
                    int orderType = -1;
                    if (Validators.isNullOrEmpty(order.order_type)) {
                        orderType = 0;
                    } else if (order.order_type.equalsIgnoreCase("delivery")) {
                        orderType = 1;
                    } else if (order.order_type.equalsIgnoreCase("pickup")) {
                        orderType = 2;
                    }
                    changeOrderStatus(orderType, order.id);
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void fetchData() {
        new Thread(this::fetchOrders).start();
    }

    private void fetchOrders() {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                    if (swipeRefreshLayout.isRefreshing()) {
                        llLoading.setVisibility(View.GONE);
                    } else {
                        llLoading.setVisibility(View.VISIBLE);
                    }

                });
            }
            HashMap<String, String> params = new HashMap<>();
            params.put("restaurant_id", myApp.getMyPreferences().getLoggedInRestaurant().id);
            isRequestOnGoing = true;
            AndroidNetworking.get(ApiEndPoints.orders + "home")
                    // .addQueryParameter("action", "StoreorderList")
                    .addQueryParameter(params)
                    .build()
                    .getAsObject(RestaurantOrderResponse.class, new ParsedRequestListener<RestaurantOrderResponse>() {
                        @Override
                        public void onResponse(RestaurantOrderResponse response) {
                            try {
                                isRequestOnGoing = false;
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        llLoading.setVisibility(View.GONE);
                                        llData.setVisibility(View.VISIBLE);
                                        swipeRefreshLayout.setRefreshing(false);
                                    });
                                }
                                ordersResponse = response;
                                setupAdapter();
                                ordersAdapter.notifyDataSetChanged();
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                if (anError.getErrorCode() == 401 || anError.getErrorCode() == 403) {
                                    myApp.getMyPreferences().logoutMerchant();
                                    myApp.getMyPreferences().logout();
                                    ToastUtils.makeLongToast(getActivity(), "Session expired");
                                    if (getActivity() != null) {
                                        new Handler(Objects.requireNonNull(Looper.myLooper()))
                                                .postDelayed(new Runnable() {
                                                    @Override
                                                    public void run() {
                                                        startActivity(new Intent(getActivity(), SplashActivity.class).setFlags(Intent.FLAG_ACTIVITY_CLEAR_TASK | Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TOP));
                                                    }
                                                }, 1500);
                                    }
                                } else {
                                    isRequestOnGoing = false;
                                    if (getActivity() != null) {
                                        getActivity().runOnUiThread(() -> {
                                            llLoading.setVisibility(View.GONE);
                                            llNoRecord.setVisibility(View.VISIBLE);
                                            swipeRefreshLayout.setRefreshing(false);
                                        });
                                    }
                                }
                                anError.printStackTrace();
                                if (!CommonFunctions.isConnected(getActivity())) {
                                    myApp.noInternet(getActivity());
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    private void fetchOrderDetail(RestaurantOrder order) {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                            progressDialog.show();
                        }
                );
            }
            AndroidNetworking.get(ApiEndPoints.orders + order.id).setPriority(Priority.IMMEDIATE)
                    .build()
                    .getAsJSONObject(new JSONObjectRequestListener() {
                        @Override
                        public void onResponse(JSONObject response) {
                            try {
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        progressDialog.dismiss();
                                    });
                                }
                                OrderDetail orderDetail = new Gson().fromJson(response.toString(), OrderDetail.class);
                                createPrint(orderDetail);
                            } catch (Exception e) {
                                e.printStackTrace();
                                CommonFunctions.showSnackBar(getActivity(), coordinatorLayout, "Order not found");
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                anError.printStackTrace();
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        progressDialog.dismiss();
                                    });
                                }
                                LogUtils.e("ERROR IN ORDER VIEW");
                                LogUtils.e(anError.getErrorBody());
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void sendRTC(String id) {
        try {
            AndroidNetworking.post(ApiEndPoints.orders + id + "/rtc").setPriority(Priority.IMMEDIATE)
                    .build()
                    .getAsJSONObject(new JSONObjectRequestListener() {
                        @Override
                        public void onResponse(JSONObject response) {
                            try {
                                CommonFunctions.showSnackBar(getActivity(), coordinatorLayout, response.getString("message"));
                            } catch (JSONException e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {

                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void changeOrderStatus(int order_type, String order_id) {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> progressDialog.show());
            }
            HashMap<String, String> params = new HashMap<>();
            params.put("status", "Delivered");
            AndroidNetworking.post(ApiEndPoints.orders + order_id + "/change-status")
                    .addQueryParameter(params)
                    .build()
                    .getAsJSONObject(new JSONObjectRequestListener() {
                        @Override
                        public void onResponse(JSONObject response) {
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> progressDialog.dismiss());
                            }
                            try {
                                OrderDetail orderDetail = new OrderDetail();
                                orderDetail.id = order_id;
                                if (order_type == 0) {
                                    orderDetail.order_type = "Dine in";
                                } else if (order_type == 1) {
                                    orderDetail.order_type = "Delivery";
                                } else if (order_type == 2) {
                                    orderDetail.order_type = "Pickup";
                                }
                                sendStatusChangeEvent("Order has been delivered", orderDetail, false);
                                fetchData();
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            anError.printStackTrace();
                            LogUtils.e("Order Status Change error");
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> progressDialog.dismiss());
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void createPrint(OrderDetail orderDetail) {
        try {
            if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains("imin")) {
                printIMin(orderDetail, false);
            } else if (myApp.getMyPreferences().getDefaultPrinter().equalsIgnoreCase(getString(R.string.ubsidi_cs30))) {
                printCs20(orderDetail, false);
            } else if (!myApp.getMyPreferences().getDefaultPrinter().equalsIgnoreCase("mobile")) {
                if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains("ip printer")) {
                    if (zoneRichPrinter == null) {
                        CommonFunctions.showSnackBar(getActivity(), coordinatorLayout, "Printer is null");
                    } else {
                        printZonerich(orderDetail, false);
                    }
                } else if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains(Constants.KP_80) || myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains(Constants.KP80)) {
                    try {
                        if (bluetoothPrinter != null) {
                            BluetoothDevice device = bluetoothPrinter.getConnectedPrinter();
                            if (device != null) {
                                CommonFunctions.functionThatDelay(200);
                                printBT(orderDetail, false);
                            } else {
                                ToastUtils.makeToast(getActivity(), "No bluetooth device found.");
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else {
                    printSunmi(orderDetail, false);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printIMin(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            myApp.iMinPrinterUtils.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    printIMin(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printZonerich(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            zoneRichPrinter.connectPrinter(myApp.getMyPreferences().getPrinterIP());
            zoneRichPrinter.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    zoneRichPrinter.connectPrinter(myApp.getMyPreferences().getPrinterIP());
                    printZonerich(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printCs20(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            cs20PrintHelper.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    printCs20(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printSunmi(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            sunmiPrinter.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    printSunmi(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void printBT(OrderDetail orderDetail, boolean kitchenCopy) {
        try {
            bluetoothPrinter.printOrder(myApp.restaurantLogo, myApp.merchantLogo, orderDetail, kitchenCopy, myApp.orderTiffinTomLogo);
            if (!kitchenCopy && !orderDetail.status.equalsIgnoreCase("failed") && !justPrint) {
                AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
                builder.setMessage("You want kitchen copy?");
                builder.setPositiveButton("Yes", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                    printBT(orderDetail, true);
                });
                builder.setNegativeButton("No", (dialogInterface, i) -> {
                    dialogInterface.dismiss();
                });
                AlertDialog alertDialog = builder.create();
                alertDialog.show();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Override
    public void onResume() {
        super.onResume();
        myApp.initService();
        if (getActivity() != null && broadcastReceiver != null)
            LocalBroadcastManager.getInstance(getActivity()).registerReceiver(broadcastReceiver, new IntentFilter(Constants.NOTIFICATION));
        if (getActivity() != null && orderStatusChangedReceiver != null)
            LocalBroadcastManager.getInstance(getActivity()).registerReceiver(orderStatusChangedReceiver, new IntentFilter(Constants.ORDER_STATUS_CHANGE));
    }

    @Override
    public void onStop() {
        if (getActivity() != null && broadcastReceiver != null)
            LocalBroadcastManager.getInstance(getActivity()).unregisterReceiver(broadcastReceiver);
        if (getActivity() != null && orderStatusChangedReceiver != null)
            LocalBroadcastManager.getInstance(getActivity()).unregisterReceiver(orderStatusChangedReceiver);
        if (getActivity() != null && progressDialog != null) {
            progressDialog.dismiss();
        }
        super.onStop();
    }

    private void sendStatusChangeEvent(String orderStatusMessage, OrderDetail orderDetail, boolean orderRejected) {
        try {
            LogUtils.e("sending status change");
            Intent pushNotification = new Intent(Constants.ORDER_STATUS_CHANGE);
            pushNotification.putExtra("rejected", orderRejected);
            pushNotification.putExtra("message", orderStatusMessage);
            pushNotification.putExtra("orderdetail", new Gson().toJson(orderDetail));
            LocalBroadcastManager.getInstance(getActivity()).sendBroadcast(pushNotification);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


}
