package com.tiffintom.partner1.fragments;

import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentTransaction;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.viewpager.widget.ViewPager;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.ParsedRequestListener;
import com.github.mikephil.charting.charts.BarChart;
import com.github.mikephil.charting.components.XAxis;
import com.github.mikephil.charting.data.BarData;
import com.github.mikephil.charting.data.BarDataSet;
import com.github.mikephil.charting.data.BarEntry;
import com.github.mikephil.charting.formatter.IndexAxisValueFormatter;
import com.github.mikephil.charting.formatter.ValueFormatter;
import com.google.android.material.button.MaterialButtonToggleGroup;
import com.lwj.widget.viewpagerindicator.ViewPagerIndicator;
import com.tiffintom.partner1.MyApp;
import com.tiffintom.partner1.R;
import com.tiffintom.partner1.adapters.RecentTransactionsAdapter;
import com.tiffintom.partner1.adapters.ScheduledPaymentViewPagerAdapter;
import com.tiffintom.partner1.base.BaseFragment;
import com.tiffintom.partner1.common.CommonFunctions;
import com.tiffintom.partner1.common.Validators;
import com.tiffintom.partner1.models.Invoice;
import com.tiffintom.partner1.models.PaymentModel;
import com.tiffintom.partner1.models.SchedulePayment;
import com.tiffintom.partner1.models.TransactionModel;
import com.tiffintom.partner1.network.ApiEndPoints;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

public class PaymentFragment extends BaseFragment {

    private ViewPager vpPayment;
    private ViewPagerIndicator vpIndicator;
    private ScheduledPaymentViewPagerAdapter vpAdapter;
    private LinearLayout llLoading, llNoRecords;
    private PaymentModel paymentModel;
    private ArrayList<TransactionModel> recentTransactions = new ArrayList<>();
    private ArrayList<SchedulePayment> schedulePayments = new ArrayList<>();
    private RecentTransactionsAdapter recentTransactionsAdapter;
    private RecyclerView rvRecentTransactions;
    private BarChart barChart;
    private MaterialButtonToggleGroup btnRange;
    private TextView tvSelectedTab, tvTotal, tvPreviousPercent;
    private LinearLayout llRestaurantRelated, llViewStatements, llTransactions;

    public static PaymentFragment getInstance() {
        return new PaymentFragment();
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {

        return inflater.inflate(R.layout.fragment_payment, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        try {
            if (loggedInRestaurant != null) {
                fetchData();
            } else {
                updateViews();
            }
            setListeners();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void manageIntents() {
        super.manageIntents();
    }

    @Override
    protected void initViews(View view) {
        super.initViews(view);
        try {
            vpPayment = view.findViewById(R.id.vpPayment);
            vpIndicator = view.findViewById(R.id.circleIndicator);
            llLoading = view.findViewById(R.id.llLoading);
            llNoRecords = view.findViewById(R.id.llNoRecord);
            rvRecentTransactions = view.findViewById(R.id.rvRecentTransactions);

            llViewStatements = view.findViewById(R.id.llViewStatements);
            llTransactions = view.findViewById(R.id.llTransactions);

            tvSelectedTab = view.findViewById(R.id.tvSelectedTab);
            tvPreviousPercent = view.findViewById(R.id.tvPreviousPercent);
            tvTotal = view.findViewById(R.id.tvTotal);

            barChart = view.findViewById(R.id.barChart);
            barChart.setDrawValueAboveBar(true);
            barChart.setFitBars(false);
            barChart.setDescription(null);
            barChart.getLegend().setEnabled(false);

            barChart.getAxisLeft().setDrawLabels(false);
            barChart.getAxisLeft().setDrawGridLines(false);
            barChart.getAxisLeft().setDrawAxisLine(false);

            barChart.getAxisRight().setDrawLabels(false);
            barChart.getAxisRight().setDrawGridLines(false);
            barChart.getAxisRight().setDrawAxisLine(false);

            barChart.getXAxis().setDrawAxisLine(false);
            barChart.getXAxis().setDrawGridLines(false);
            barChart.getXAxis().setGranularity(1);
            barChart.getXAxis().setPosition(XAxis.XAxisPosition.BOTTOM);
            barChart.setDoubleTapToZoomEnabled(false);
            barChart.setDoubleTapToZoomEnabled(false);
            barChart.setPinchZoom(false);
            barChart.setScaleEnabled(false);

            btnRange = view.findViewById(R.id.btnRangeGroup);

            vpAdapter = new ScheduledPaymentViewPagerAdapter(getActivity(), schedulePayments);
            vpPayment.setAdapter(vpAdapter);
            vpIndicator.setViewPager(vpPayment);

            rvRecentTransactions.setLayoutManager(new LinearLayoutManager(getActivity(), RecyclerView.VERTICAL, false));
            recentTransactionsAdapter = new RecentTransactionsAdapter(getActivity(), recentTransactions, (view1, position, data) -> {

            });
            rvRecentTransactions.setAdapter(recentTransactionsAdapter);

            llRestaurantRelated = view.findViewById(R.id.llRestaurantRelated);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private void setListeners() {

        try {
            llViewStatements.setOnClickListener(v -> {
                FragmentTransaction ft = getActivity().getSupportFragmentManager().beginTransaction();
                ft.replace(R.id.nav_host_fragment, PaymentStatementsFragment.getInstance(), "Statements");
                ft.addToBackStack("can_go_back");
                ft.commit();
            });

            llTransactions.setOnClickListener(v -> {
                FragmentTransaction ft = getActivity().getSupportFragmentManager().beginTransaction();
                ft.replace(R.id.nav_host_fragment, PaymentTransactionFragment.getInstance(), "Transactions");
                ft.addToBackStack("can_go_back");
                ft.commit();
            });

            btnRange.addOnButtonCheckedListener((group, checkedId, isChecked) -> {
                if (isChecked) {

                    if (checkedId == R.id.btnThisMonth) {
                        setUpThisMonthChart();
                    } else if (checkedId == R.id.btnLastMonth) {
                        setUpLastMonthChart();
                    } else if (checkedId == R.id.btnThisYear) {
                        setUpThisYearChart();
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    private void updateViews() {
        try {
            if (loggedInRestaurant == null) {
                llLoading.setVisibility(View.GONE);
                llRestaurantRelated.setVisibility(View.GONE);
            }
            schedulePayments.clear();
            if (paymentModel != null) {
                if (!Validators.isNullOrEmpty(paymentModel.next_schedule_date)) {
                    SchedulePayment schedulePayment = new SchedulePayment();
                    schedulePayment.title = "Next scheduled payment";
                    schedulePayment.scheduleDate = paymentModel.next_schedule_date;
                    schedulePayment.weekFrom = paymentModel.for_week_next_start;
                    schedulePayment.weekTo = paymentModel.for_week_next_end;
                    schedulePayments.add(schedulePayment);
                }
                if (!Validators.isNullOrEmpty(paymentModel.latest_schedule_date)) {
                    SchedulePayment schedulePayment = new SchedulePayment();
                    schedulePayment.title = "Latest scheduled payment";
                    schedulePayment.scheduleDate = paymentModel.latest_schedule_date;
                    schedulePayment.weekFrom = paymentModel.for_week_latest_start;
                    schedulePayment.weekTo = paymentModel.for_week_latest_end;
                    schedulePayments.add(schedulePayment);
                }
                vpAdapter.notifyDataSetChanged();
                vpIndicator.setViewPager(vpPayment, schedulePayments.size());
                recentTransactionsAdapter.notifyDataSetChanged();

                int checkedId = btnRange.getCheckedButtonId();

                if (checkedId == R.id.btnThisMonth) {
                    setUpThisMonthChart();
                } else if (checkedId == R.id.btnLastMonth) {
                    setUpLastMonthChart();
                } else if (checkedId == R.id.btnThisYear) {
                    setUpThisYearChart();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    private void fetchData() {
        new Thread(this::fetchPaymentOverview).start();
    }

    private void fetchPaymentOverview() {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                    llLoading.setVisibility(View.VISIBLE);
                });
            }
            HashMap<String, String> params = new HashMap<>();
            params.put("restaurant_id", myApp.getMyPreferences().getLoggedInRestaurant().id);
            AndroidNetworking.get(ApiEndPoints.payment_overview)
                    .addQueryParameter(params)
                    .build()
                    .getAsObject(PaymentModel.class, new ParsedRequestListener<PaymentModel>() {
                        @Override
                        public void onResponse(PaymentModel response) {
                            try {
                                paymentModel = response;
                                if (paymentModel.payment_transactions != null) {
                                    recentTransactions.clear();
                                    recentTransactions.addAll(paymentModel.payment_transactions);
                                }
                                updateViews();
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        llLoading.setVisibility(View.GONE);
                                    });
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            anError.printStackTrace();
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> {
                                    llLoading.setVisibility(View.GONE);
                                    llNoRecords.setVisibility(View.VISIBLE);
                                });
                            }

                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private void setUpThisMonthChart() {
        try {
            List<BarEntry> barEntries = new ArrayList<>();
            float count = 0;
            ArrayList<String> dates = new ArrayList<>();
            for (Invoice invoice : paymentModel.this_month_invoices) {
                barEntries.add(new BarEntry(count, invoice.grand_total));
                if (!Validators.isNullOrEmpty(invoice.end_date))
                    dates.add(CommonFunctions.formatUnknownDateTime(invoice.end_date, MyApp.DEL_TIMEFORMAT, "dd/MM"));
                else if (!Validators.isNullOrEmpty(invoice.start_date)) {
                    dates.add(CommonFunctions.formatUnknownDateTime(invoice.start_date, MyApp.DEL_TIMEFORMAT, "dd/MM"));
                } else {
                    dates.add("01/01");
                }
                count++;
            }
            BarDataSet barDataSet = new BarDataSet(barEntries, "Dates");
            barDataSet.setColor(ContextCompat.getColor(getActivity(), R.color.persian_green));
            barDataSet.setValueFormatter(new ValueFormatter() {
                @Override
                public String getFormattedValue(float value) {
                    return (value == 0) ? "" : myApp.getCurrencySymbol() + value;
                }
            });
            barDataSet.setValueTextSize(12);
            BarData barData = new BarData(barDataSet);
            barData.setBarWidth(0.9f);
            barChart.setData(barData);

            XAxis xAxis = barChart.getXAxis();
            xAxis.setValueFormatter(new IndexAxisValueFormatter(dates));
            barChart.invalidate();

            tvPreviousPercent.setVisibility(View.VISIBLE);
            tvSelectedTab.setText("This Month");
            tvPreviousPercent.setText(paymentModel.this_month_percentage + "% from previous");
            tvTotal.setText(myApp.getCurrencySymbol() + MyApp.df.format(Float.parseFloat(paymentModel.this_month_grand_total)));
        } catch (NumberFormatException e) {
            e.printStackTrace();
        }
    }

    private void setUpLastMonthChart() {
        try {
            List<BarEntry> barEntries = new ArrayList<>();
            ArrayList<String> dates = new ArrayList<>();
            int count = 0;
            for (Invoice invoice : paymentModel.previous_month_invoices) {
                barEntries.add(new BarEntry(count, invoice.grand_total));
                if (!Validators.isNullOrEmpty(invoice.end_date))
                    dates.add(CommonFunctions.formatUnknownDateTime(invoice.end_date, MyApp.DEL_TIMEFORMAT, "dd/MM"));
                else if (!Validators.isNullOrEmpty(invoice.start_date)) {
                    dates.add(CommonFunctions.formatUnknownDateTime(invoice.start_date, MyApp.DEL_TIMEFORMAT, "dd/MM"));
                } else {
                    dates.add("01/01");
                }
                count++;
            }
            BarDataSet barDataSet = new BarDataSet(barEntries, "Dates");
            barDataSet.setColor(ContextCompat.getColor(getActivity(), R.color.persian_green));
            barDataSet.setValueFormatter(new ValueFormatter() {
                @Override
                public String getFormattedValue(float value) {
                    return (value == 0) ? "" : myApp.getCurrencySymbol() + value;
                }
            });
            barDataSet.setValueTextSize(12);
            BarData barData = new BarData(barDataSet);
            barData.setBarWidth(0.9f);
            barChart.setData(barData);

            XAxis xAxis = barChart.getXAxis();
            xAxis.setValueFormatter(new IndexAxisValueFormatter(dates));
            barChart.invalidate();

            tvPreviousPercent.setVisibility(View.GONE);
            tvSelectedTab.setText("Last Month");
            tvTotal.setText(myApp.getCurrencySymbol() + MyApp.df.format(Float.parseFloat(paymentModel.previous_month_grand_total)));
        } catch (NumberFormatException e) {
            e.printStackTrace();
        }
    }

    private void setUpThisYearChart() {
        try {
            List<BarEntry> barEntries = new ArrayList<>();
            ArrayList<String> dates = new ArrayList<String>();
            int count = 0;
            for (Invoice invoice : paymentModel.this_year_invoices) {
                barEntries.add(new BarEntry(count, invoice.grand_total));
                if (!Validators.isNullOrEmpty(invoice.end_date))
                    dates.add(CommonFunctions.formatUnknownDateTime(invoice.end_date, MyApp.DEL_TIMEFORMAT, "dd/MM"));
                else if (!Validators.isNullOrEmpty(invoice.start_date)) {
                    dates.add(CommonFunctions.formatUnknownDateTime(invoice.start_date, MyApp.DEL_TIMEFORMAT, "dd/MM"));
                } else {
                    dates.add("01/01");
                }
                count++;
            }
            BarDataSet barDataSet = new BarDataSet(barEntries, "Dates");
            barDataSet.setColor(ContextCompat.getColor(getActivity(), R.color.persian_green));
            barDataSet.setValueFormatter(new ValueFormatter() {
                @Override
                public String getFormattedValue(float value) {
                    return (value == 0) ? "" : myApp.getCurrencySymbol() + value;
                }
            });
            barDataSet.setValueTextSize(12);
            BarData barData = new BarData(barDataSet);
            barData.setBarWidth(0.9f);
            barChart.setData(barData);

            XAxis xAxis = barChart.getXAxis();
            xAxis.setValueFormatter(new IndexAxisValueFormatter(dates));
            barChart.invalidate();

            tvPreviousPercent.setVisibility(View.GONE);
            tvSelectedTab.setText("This Year");
            tvTotal.setText(myApp.getCurrencySymbol() + MyApp.df.format(Float.parseFloat(paymentModel.this_year_grand_total)));
        } catch (NumberFormatException e) {
            e.printStackTrace();
        }
    }


    @Override
    public void onStop() {
        super.onStop();
    }
}
