package com.tiffintom.partner1.fragments;

import android.app.DatePickerDialog;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.cardview.widget.CardView;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.JSONObjectRequestListener;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.tiffintom.partner1.MyApp;
import com.tiffintom.partner1.R;
import com.tiffintom.partner1.adapters.TransactionsAndStatementsAdapter;
import com.tiffintom.partner1.base.BaseFragment;
import com.tiffintom.partner1.common.CommonFunctions;
import com.tiffintom.partner1.common.StickHeaderItemDecoration;
import com.tiffintom.partner1.common.Validators;
import com.tiffintom.partner1.models.Header;
import com.tiffintom.partner1.models.PaymentLink;
import com.tiffintom.partner1.network.ApiEndPoints;
import com.tiffintom.partner1.utils.LogUtils;

import org.json.JSONObject;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;

public class PaymentLinksTransactionFragment extends BaseFragment {

    private final ArrayList<Object> transactions = new ArrayList<>();
    private RecyclerView rvTransactions;
    private TransactionsAndStatementsAdapter transactionsAndStatementsAdapter;
    private SwipeRefreshLayout swipeRefreshLayout;
    private String fromDate = "2019-06-01", toDate = "2020-12-31";
    private final Calendar today = Calendar.getInstance();
    private final Calendar fromCalendar = Calendar.getInstance();
    private final Calendar toCalendar = Calendar.getInstance();
    private final ArrayList<PaymentLink> paymentLinks = new ArrayList<>();
    private TextView tvToDate, tvFromDate, tvSelectTransactions;
    private RelativeLayout llFromDate, llToDate;
    int fromYear, fromMonth, fromDay, toYear, toMonth, toDay;
    private TextView tvTitle;
    private boolean isLoading = false;
    private int currentPage = 1, limit = 10;
    private ImageView ivBack;
    private CardView cvCalender;
    private LinearLayout llMainLayout;

    public static PaymentLinksTransactionFragment getInstance() {
        return new PaymentLinksTransactionFragment();
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragement_transaction, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        try {
            updateViews();
            setListeners();
            getData();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void manageIntents() {
        super.manageIntents();
    }

    @Override
    protected void initViews(View view) {
        super.initViews(view);
        try {
            llMainLayout = view.findViewById(R.id.llMainLayout);
            tvSelectTransactions = view.findViewById(R.id.tvSelectTransactions);
            tvTitle = view.findViewById(R.id.tvTitle);
            tvSelectTransactions.setVisibility(View.GONE);

            tvToDate = view.findViewById(R.id.tvToDate);
            tvFromDate = view.findViewById(R.id.tvFromDate);
            llToDate = view.findViewById(R.id.llToDate);
            llFromDate = view.findViewById(R.id.llFromDate);
            ivBack = view.findViewById(R.id.ivBack);
            cvCalender = view.findViewById(R.id.cvCalender);
            cvCalender.setVisibility(View.GONE);
            rvTransactions = view.findViewById(R.id.rvTransactions);
            swipeRefreshLayout = view.findViewById(R.id.swipeRefreshLayout);
            transactionsAndStatementsAdapter = new TransactionsAndStatementsAdapter(getActivity(), transactions, (position, data) -> resendPaymentLink((PaymentLink) data));
            rvTransactions.setLayoutManager(new LinearLayoutManager(getActivity(), RecyclerView.VERTICAL, false));
            rvTransactions.setNestedScrollingEnabled(false);
            rvTransactions.setAdapter(transactionsAndStatementsAdapter);
            rvTransactions.addItemDecoration(new StickHeaderItemDecoration(transactionsAndStatementsAdapter));

            tvTitle.setText("Payment link transactions");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void updateViews() {
        try {
            fromCalendar.add(Calendar.DAY_OF_MONTH, -1);

            fromDate = CommonFunctions.formatMiliToDesireFormat(fromCalendar.getTimeInMillis(), "yyyy-MM-dd");
            toDate = CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd");

            tvToDate.setText("To: " + CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "dd/MM/yyyy"));
            tvFromDate.setText("From: " + CommonFunctions.formatMiliToDesireFormat(fromCalendar.getTimeInMillis(), "dd/MM/yyyy"));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void setListeners() {
        try {
            ivBack.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    getActivity().onBackPressed();
                }
            });
            rvTransactions.addOnScrollListener(new RecyclerView.OnScrollListener() {
                @Override
                public void onScrollStateChanged(@NonNull RecyclerView recyclerView, int newState) {
                    super.onScrollStateChanged(recyclerView, newState);
                }

                @Override
                public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                    super.onScrolled(recyclerView, dx, dy);

                    LinearLayoutManager linearLayoutManager = (LinearLayoutManager) recyclerView.getLayoutManager();
                    LogUtils.e("Scrolled");
                    if (!isLoading) {
                        if (linearLayoutManager != null && linearLayoutManager.findLastCompletelyVisibleItemPosition() == transactions.size() - 1) {
                            //bottom of list!
                            LogUtils.e("Scrolled at bottom");
                            currentPage++;
                            fetchTransactions();
                            isLoading = true;
                        }
                    }
                }
            });

            swipeRefreshLayout.setOnRefreshListener(() -> {
                currentPage = 1;
                getData();
            });
            llFromDate.setOnClickListener(view -> {
                DatePickerDialog datePickerDialog = new DatePickerDialog(getActivity(), (datePicker, y, m, d) -> {
                    fromDay = d;
                    fromMonth = m;
                    fromYear = y;
                    fromDate = y + "-" + String.format("%02d", (m + 1)) + "-" + String.format("%02d", d);
                    fromCalendar.set(y, m, d);
                    tvFromDate.setText("From: " + CommonFunctions.formatMiliToDesireFormat(fromCalendar.getTimeInMillis(), "dd/MM/yyyy"));
                    currentPage = 1;
                    fetchTransactions();
                }, fromCalendar.get(Calendar.YEAR), fromCalendar.get(Calendar.MONTH), fromCalendar.get(Calendar.DAY_OF_MONTH));
                datePickerDialog.show();
                if (!Validators.isNullOrEmpty(toDate)) {
                    datePickerDialog.getDatePicker().setMaxDate(toCalendar.getTimeInMillis());
                }

            });
            llToDate.setOnClickListener(view -> {
                DatePickerDialog datePickerDialog = new DatePickerDialog(getActivity(), (datePicker, y, m, d) -> {
                    toDay = d;
                    toMonth = m;
                    toYear = y;
                    toDate = y + "-" + String.format("%02d", (m + 1)) + "-" + String.format("%02d", d);
                    toCalendar.set(y, m, d);
                    tvToDate.setText("To: " + CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "dd/MM/yyyy"));
                    currentPage = 1;
                    fetchTransactions();
                }, toCalendar.get(Calendar.YEAR), toCalendar.get(Calendar.MONTH), toCalendar.get(Calendar.DAY_OF_MONTH));
                datePickerDialog.show();
                if (!Validators.isNullOrEmpty(fromDate)) {
                    datePickerDialog.getDatePicker().setMinDate(fromCalendar.getTimeInMillis());
                }

            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private void getData() {
        new Thread(this::fetchTransactions).start();
    }

    Type paymentLinkTypeToken = new TypeToken<List<PaymentLink>>() {
    }.getType();

    private void fetchTransactions() {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                    swipeRefreshLayout.setRefreshing(true);
                });
            }
            HashMap<String, String> params = new HashMap<>();
            params.put("to_date", CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd").replace("/", "-"));
            params.put("from_date", CommonFunctions.getPreviousDate(CommonFunctions.formatMiliToDesireFormat(toCalendar.getTimeInMillis(), "yyyy-MM-dd").replace("/", "-"), -2));
            params.put("business_id", merchantBusiness.id);
            params.put("page", String.valueOf(currentPage));
            params.put("per_page", String.valueOf(limit));
            AndroidNetworking.get(ApiEndPoints.payment_message)
                    .addQueryParameter(params)
                    .build()
                    .getAsJSONObject(new JSONObjectRequestListener() {
                        @Override
                        public void onResponse(JSONObject response) {
                            isLoading = false;
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> {
                                    swipeRefreshLayout.setRefreshing(false);
                                });
                            }
                            try {
                                if (currentPage == 1)
                                    paymentLinks.clear();
                                paymentLinks.addAll(new Gson().fromJson(response.getJSONArray("data").toString(), paymentLinkTypeToken));
                                setupAdapter();
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                anError.printStackTrace();
                                isLoading = false;
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        swipeRefreshLayout.setRefreshing(false);
                                    });
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void resendPaymentLink(PaymentLink data) {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> progressDialog.show());
            }
            AndroidNetworking.post(ApiEndPoints.payment_message + "/" + data.id + "/resend")
                    .build()
                    .getAsJSONObject(new JSONObjectRequestListener() {
                        @Override
                        public void onResponse(JSONObject response) {
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> {
                                    progressDialog.dismiss();
                                });
                            }
                            try {
                                CommonFunctions.showSnackBar(getActivity(), llMainLayout, "Message sent successfully.");
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> progressDialog.dismiss());
                            }
                            anError.printStackTrace();
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private final ArrayList<String> dayMonth = new ArrayList<>();

    private void setupAdapter() {
        try {
            dayMonth.clear();
            Calendar fromCalendarTemp = Calendar.getInstance();
            Calendar toCalendarTemp = Calendar.getInstance();

            fromCalendarTemp.setTime(fromCalendar.getTime());
            toCalendarTemp.setTime(toCalendar.getTime());

            fromCalendarTemp.add(Calendar.DAY_OF_MONTH, -1);
            while (toCalendarTemp.after(fromCalendarTemp)) {
                String dMonth = CommonFunctions.formatMiliToDesireFormat(toCalendarTemp.getTimeInMillis(), "dd MMMM");
                dayMonth.add(dMonth);
                toCalendarTemp.add(Calendar.DAY_OF_MONTH, -1);
            }

            transactions.clear();

            for (String ddMMM : dayMonth) {
                ArrayList<PaymentLink> dailyTransactions = new ArrayList<>();
                for (PaymentLink transactionModel : paymentLinks) {
                    String transactionDateMonth = CommonFunctions.formatUnknownDateTime(transactionModel.created, MyApp.PHP_DATE_TIME_FORMAT_ZULU, "dd MMMM");
                    if (transactionDateMonth.equals(ddMMM)) {
                        dailyTransactions.add(transactionModel);
                    }
                }
                if (dailyTransactions.size() > 0) {
                    transactions.add(new Header(CommonFunctions.formatUnknownDateTime(ddMMM, "dd MMMM", "EEE dd MMM")));
                    transactions.addAll(dailyTransactions);
                }
            }
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                    transactionsAndStatementsAdapter.notifyDataSetChanged();
                });
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
