package com.tiffintom.partner1.fragments;

import android.bluetooth.BluetoothDevice;
import android.content.res.Resources;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CompoundButton;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.SwitchCompat;
import androidx.coordinatorlayout.widget.CoordinatorLayout;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentTransaction;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.common.ANRequest;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.JSONObjectRequestListener;
import com.androidnetworking.interfaces.ParsedRequestListener;
import com.google.android.material.snackbar.Snackbar;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.tiffintom.partner1.R;
import com.tiffintom.partner1.adapters.ReservationsAdapter;
import com.tiffintom.partner1.base.BaseFragment;
import com.tiffintom.partner1.common.CommonFunctions;
import com.tiffintom.partner1.common.StickHeaderItemDecoration;
import com.tiffintom.partner1.common.Validators;
import com.tiffintom.partner1.common.bluetoothprinter.BluetoothPrinter;
import com.tiffintom.partner1.common.printer.SunmiPrinter;
import com.tiffintom.partner1.common.printer.ZoneRichPrinter;
import com.tiffintom.partner1.interfaces.DialogDismissListener;
import com.tiffintom.partner1.models.OrderStatusIndicatorHeader;
import com.tiffintom.partner1.models.Reservation;
import com.tiffintom.partner1.models.RestaurantOrderResponse;
import com.tiffintom.partner1.network.ApiEndPoints;
import com.tiffintom.partner1.utils.Constants;
import com.tiffintom.partner1.utils.LogUtils;
import com.tiffintom.partner1.utils.ToastUtils;

import org.json.JSONObject;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;

/**
 * Created by Amrish on 09-11-2020.
 */
public class ReservationsFragment extends BaseFragment {
    private RestaurantOrderResponse ordersResponse;
    private ArrayList<Object> reservationsList = new ArrayList<>();
    private ArrayList<Reservation> reservationsOriginal = new ArrayList<>();
    private RecyclerView rvOrders;
    private SwitchCompat switchCompat;
    private ReservationsAdapter reservationsAdapter;
    private Type reservationType = new TypeToken<List<Reservation>>() {
    }.getType();
    private LinearLayout llLoading, llNoRecord;
    private CoordinatorLayout coordinatorLayout;
    private SwipeRefreshLayout swipeRefreshLayout;
    private SunmiPrinter sunmiPrinter;
    public static ZoneRichPrinter zoneRichPrinter;
    private BluetoothPrinter bluetoothPrinter;
    private TextView tvViewAll;
    private Snackbar snackbar, greySnackBar;
    private final Calendar toCalendar = Calendar.getInstance();

    public static ReservationsFragment getInstance() {
        return new ReservationsFragment();
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_reservations, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        try {
            updateOrderTakingStatus();
            setListeners();
            fetchData();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void manageIntents() {
        super.manageIntents();
    }

    int id;

    @Override
    protected void initViews(View view) {
        super.initViews(view);
        try {
            sunmiPrinter = new SunmiPrinter(getActivity());
            if (!Validators.isNullOrEmpty(myApp.getMyPreferences().getPrinterIP())) {
                zoneRichPrinter = new ZoneRichPrinter(getActivity());
            }
            bluetoothPrinter = new BluetoothPrinter(getActivity());
            rvOrders = view.findViewById(R.id.rvReservations);
            switchCompat = view.findViewById(R.id.switchReservations);
            llLoading = view.findViewById(R.id.llLoading);
            llNoRecord = view.findViewById(R.id.llNoRecord);
            tvViewAll = view.findViewById(R.id.tvViewAll);
            swipeRefreshLayout = view.findViewById(R.id.swipeRefreshLayout);
            coordinatorLayout = view.findViewById(R.id.coordinatorLayout);

            reservationsAdapter = new ReservationsAdapter(getActivity(), reservationsList, (view1, position, data) -> {
                //View clicked
                Reservation reservation = (Reservation) data;
                id = reservation.id;
                ReservationViewBottomSheetFragment reservationViewBottomSheetFragment = ReservationViewBottomSheetFragment.getInstance((Reservation) data);
                reservationViewBottomSheetFragment.show(getChildFragmentManager(), "reservation_view");
                reservationViewBottomSheetFragment.setDialogDismissListener(reservationViewDismissListener);
            }, (view1, position, data) -> {
                //Accept clicked
                Reservation reservation = (Reservation) data;
                id = reservation.id;
                openConfirmReservationDialog();

            }, (view12, position, data) -> {
                //Cancel clicked
                Reservation reservation = (Reservation) data;
                id = reservation.id;
                openRejectReservationDialog();

            }, (view1, position, data) -> {
                //Print clicked
                Reservation printReservation = (Reservation) data;
                if (!myApp.getMyPreferences().getDefaultPrinter().equalsIgnoreCase("mobile")) {
                    if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains("imin")) {
                        myApp.iMinPrinterUtils.printReservation(printReservation, myApp.restaurantLogo);
                    } else if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains("ip printer")) {
                        if (zoneRichPrinter == null) {
                            CommonFunctions.showSnackBar(getActivity(), coordinatorLayout, "Printer is not connected");
                        } else {
                            zoneRichPrinter.connectPrinter(myApp.getMyPreferences().getPrinterIP());
                            zoneRichPrinter.printReservation(printReservation, myApp.restaurantLogo);
                        }
                    } else if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains(Constants.KP_80) || myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains(Constants.KP80)) {
                        try {
                            if (bluetoothPrinter != null) {
                                BluetoothDevice device = bluetoothPrinter.getConnectedPrinter();
                                if (device != null) {
                                    CommonFunctions.functionThatDelay(200);
                                    bluetoothPrinter.printReservation(printReservation, myApp.restaurantLogo);
                                } else {
                                    ToastUtils.makeToast(getActivity(), "No bluetooth device found.");
                                }
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    } else {
                        sunmiPrinter.printReservation(getActivity(), printReservation);
                    }
                }
            });
            rvOrders.setLayoutManager(new LinearLayoutManager(getActivity(), RecyclerView.VERTICAL, false));
            rvOrders.setNestedScrollingEnabled(false);
            rvOrders.addItemDecoration(new StickHeaderItemDecoration(reservationsAdapter));
            rvOrders.setAdapter(reservationsAdapter);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void openConfirmReservationDialog() {
        try {
            ConfirmationDialogFragment confirmationDialogFragment = ConfirmationDialogFragment.getInstance("Accept Booking", "Do you want to confirm this booking?", false);
            confirmationDialogFragment.show(getChildFragmentManager(), "reservation_accept");
            confirmationDialogFragment.setDialogDismissListener(o -> {
                String action = (String) o; //confirm, cancel
                if (action.equalsIgnoreCase("confirm")) {
                    changeStatus("Approved");
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void setListeners() {
        try {
            switchCompat.setOnCheckedChangeListener(checkedChangeListener);
            swipeRefreshLayout.setOnRefreshListener(this::fetchData);
            tvViewAll.setOnClickListener(view -> {
                FragmentTransaction ft = getActivity().getSupportFragmentManager().beginTransaction();
                ft.replace(R.id.nav_host_fragment, ReservationsAllFragment.getInstance(), "all_reservations");
                ft.addToBackStack("let_go_back");
                ft.commit();
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    CompoundButton.OnCheckedChangeListener checkedChangeListener = new CompoundButton.OnCheckedChangeListener() {
        @Override
        public void onCheckedChanged(CompoundButton compoundButton, boolean b) {
            try {
                EnterPasswordDialogFragment enterPasswordDialogFragment = EnterPasswordDialogFragment.getInstance("booking_status", b ? "Yes" : "No");
                enterPasswordDialogFragment.show(getChildFragmentManager(), "enter_password");
                enterPasswordDialogFragment.setCancelable(false);
                enterPasswordDialogFragment.setDialogDismissListener(changeOrderAcceptingStatusDialogListener);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    };
    DialogDismissListener changeOrderAcceptingStatusDialogListener = new DialogDismissListener() {
        @Override
        public void onDialogDismiss(Object o) {
            switchCompat.setOnCheckedChangeListener(null);
            updateOrderTakingStatus();
            switchCompat.setOnCheckedChangeListener(checkedChangeListener);
        }
    };
    DialogDismissListener reservationViewDismissListener = o -> {
        String action = (String) o;
        if (action.equalsIgnoreCase("accept")) {
            openConfirmReservationDialog();
        } else if (action.equalsIgnoreCase("reject")) {
            openRejectReservationDialog();
        }
    };
    String cancelReason = "";

    private void openRejectReservationDialog() {
        try {
            RejectReasonDialogFragment rejectReasonDialogFragment = RejectReasonDialogFragment.getInstance("Customer cancelled,Fully booked,Customer error,Other");
            rejectReasonDialogFragment.show(getChildFragmentManager(), "reject_reservation");
            rejectReasonDialogFragment.setDialogDismissListener(o -> {
                cancelReason = (String) o;
                changeStatus("Cancel");
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void updateOrderTakingStatus() {
        switchCompat.setChecked(myApp.getMyPreferences().getLoggedInRestaurant().booking_status.equalsIgnoreCase("Yes"));
    }

    private void fetchData() {
        new Thread(this::fetchOrders).start();
    }

    private void fetchOrders() {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                    if (swipeRefreshLayout.isRefreshing()) {
                        llLoading.setVisibility(View.GONE);
                    } else {
                        llLoading.setVisibility(View.VISIBLE);
                    }
                });
            }
            HashMap<String, String> params = new HashMap<>();
            params.put("restaurant_id", myApp.getMyPreferences().getLoggedInRestaurant().id);
            AndroidNetworking.get(ApiEndPoints.bookings + "/home")
                    .addQueryParameter(params)
                    .build()
                    .getAsJSONObject(new JSONObjectRequestListener() {
                        @Override
                        public void onResponse(JSONObject response) {
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> {
                                    llLoading.setVisibility(View.GONE);
                                    swipeRefreshLayout.setRefreshing(false);
                                });
                            }
                            reservationsOriginal.clear();
                            reservationsList.clear();
                            try {
                                reservationsOriginal.addAll(new Gson().fromJson(response.getJSONArray("pending_bookings").toString(), reservationType));
                                sortReservations();
                                if (reservationsOriginal.size() == 0)
                                    reservationsAdapter.notifyDataSetChanged();
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> {
                                    llLoading.setVisibility(View.GONE);
                                    llNoRecord.setVisibility(View.VISIBLE);
                                    swipeRefreshLayout.setRefreshing(false);
                                });
                            }
                            LogUtils.e("FETCH RESERVATIONS ERROR");
                            anError.printStackTrace();
                            if (!CommonFunctions.isConnected(getActivity())) {
                                myApp.noInternet(getActivity());
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    private void sortReservations() {
        try {
            ArrayList<Reservation> newBookings = new ArrayList<>();
            ArrayList<Reservation> inProgressBookings = new ArrayList<>();
            ArrayList<Reservation> completedBookings = new ArrayList<>();

            Calendar todayCalendar = Calendar.getInstance();
            todayCalendar.add(Calendar.DAY_OF_MONTH, -1);
            for (Reservation reservation : reservationsOriginal) {
                if (reservation.status.equalsIgnoreCase("pending")) {
                    newBookings.add(reservation);
                } else if (CommonFunctions.convertStringDateToDate(reservation.booking_date, "yyyy-MM-dd").after(todayCalendar.getTime()) && reservation.status.equalsIgnoreCase("approved")) {
                    inProgressBookings.add(reservation);
                } else if (reservation.status.equalsIgnoreCase("cancel") || CommonFunctions.convertStringDateToDate(reservation.booking_date, "yyyy-MM-dd").before(todayCalendar.getTime())) {
                    completedBookings.add(reservation);
                }
            }
            reservationsList.clear();
            if (newBookings.size() > 0) {
                reservationsList.add(new OrderStatusIndicatorHeader("New booking (" + newBookings.size() + ")"));
                reservationsList.addAll(newBookings);
            }
            if (inProgressBookings.size() > 0) {
                reservationsList.add(new OrderStatusIndicatorHeader("Booking in progress (" + inProgressBookings.size() + ")"));
                reservationsList.addAll(inProgressBookings);
            }
            if (completedBookings.size() > 0) {
                reservationsList.add(new OrderStatusIndicatorHeader("Completed bookings (" + completedBookings.size() + ")"));
                reservationsList.addAll(completedBookings);
            }
            reservationsAdapter.notifyDataSetChanged();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private void changeStatus(String status) {
        try {
            if (getActivity() != null) {
                getActivity().runOnUiThread(() -> {
                    progressDialog.show();
                });
            }
            HashMap<String, String> params = new HashMap<>();
            params.put("restaurant_id", myApp.getMyPreferences().getLoggedInRestaurant().id);

            if (status.equalsIgnoreCase("Cancel")) {
                params.put("cancel_reason", cancelReason);
            }
            params.put("status", status);
            ANRequest.PostRequestBuilder anRequest;
            anRequest = AndroidNetworking.post(ApiEndPoints.bookings + "/" + id + "/change-status")
                    .addQueryParameter(params);
            anRequest
                    .build()
                    .getAsObject(Reservation.class, new ParsedRequestListener<Reservation>() {
                        @Override
                        public void onResponse(Reservation response) {
                            try {
                                if (getActivity() != null) {
                                    getActivity().runOnUiThread(() -> {
                                        progressDialog.dismiss();
                                    });
                                }
                                Reservation printReservation = response;
                                cancelReason = "";
                                if (!myApp.getMyPreferences().getDefaultPrinter().equalsIgnoreCase("mobile")) {
                                    if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains("imin")) {
                                        myApp.iMinPrinterUtils.printReservation(printReservation, myApp.restaurantLogo);
                                    } else if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains("ip printer")) {
                                        if (zoneRichPrinter == null) {
                                            CommonFunctions.showSnackBar(getActivity(), coordinatorLayout, "Printer is not connected");
                                        } else {
                                            zoneRichPrinter.connectPrinter(myApp.getMyPreferences().getPrinterIP());
                                            zoneRichPrinter.printReservation(printReservation, myApp.restaurantLogo);
                                        }
                                    } else if (myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains(Constants.KP_80) || myApp.getMyPreferences().getDefaultPrinter().toLowerCase().contains(Constants.KP80)) {
                                        try {
                                            if (bluetoothPrinter != null) {
                                                BluetoothDevice device = bluetoothPrinter.getConnectedPrinter();
                                                if (device != null) {
                                                    CommonFunctions.functionThatDelay(200);
                                                    bluetoothPrinter.printReservation(printReservation, myApp.restaurantLogo);
                                                } else {
                                                    ToastUtils.makeToast(getActivity(), "No bluetooth device found.");
                                                }
                                            }
                                        } catch (Exception e) {
                                            e.printStackTrace();
                                        }
                                    } else {
                                        sunmiPrinter.printReservation(getActivity(), printReservation);
                                    }
                                }
                                showSnackBar(status, printReservation);
                                fetchData();

                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            anError.printStackTrace();
                            if (getActivity() != null) {
                                getActivity().runOnUiThread(() -> {
                                    progressDialog.dismiss();
                                });
                            }
                            if (!CommonFunctions.isConnected(getActivity())) {
                                myApp.noInternet(getActivity());
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void showSnackBar(String status, Reservation reservation) {
        try {
            snackbar = Snackbar.make(coordinatorLayout, "Reservation status has been changed", Snackbar.LENGTH_LONG)
                    .setActionTextColor(getResources().getColor(R.color.white))
                    .setAction("view", v -> {
                        ReservationViewBottomSheetFragment reservationViewBottomSheetFragment = ReservationViewBottomSheetFragment.getInstance(reservation);
                        reservationViewBottomSheetFragment.show(getChildFragmentManager(), "reservation_view");
                        reservationViewBottomSheetFragment.setDialogDismissListener(o -> fetchData());
                    });
            if (status.equalsIgnoreCase("cancel")) {
                snackbar.setBackgroundTint(ContextCompat.getColor(getActivity(), R.color.orders_card_bg_orange));
            } else {
                snackbar.setBackgroundTint(ContextCompat.getColor(getActivity(), R.color.orders_card_bg_green));
            }
            snackbar.show();
        } catch (Resources.NotFoundException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onStop() {
        try {
            if (getActivity() != null && progressDialog != null) {
                progressDialog.dismiss();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        super.onStop();
    }

}
