package com.tiffintom.partner1.fragments;

import android.content.res.Resources;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.AppCompatEditText;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.ParsedRequestListener;
import com.google.android.material.button.MaterialButton;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.iarcuschin.simpleratingbar.SimpleRatingBar;
import com.tiffintom.partner1.R;
import com.tiffintom.partner1.base.BaseBottomSheet;
import com.tiffintom.partner1.common.CommonFunctions;
import com.tiffintom.partner1.common.Validators;
import com.tiffintom.partner1.interfaces.DialogDismissListener;
import com.tiffintom.partner1.models.LatestReviewModel;
import com.tiffintom.partner1.models.ReviewModel;
import com.tiffintom.partner1.network.ApiEndPoints;

import java.util.HashMap;

public class ReviewResponseFragment extends BaseBottomSheet {

    private DialogDismissListener dialogDismissListener;
    private LatestReviewModel reviewModel;
    private TextView tvName, tvDate, tvRating, tvMessage, tvWordCounter;
    private AppCompatEditText etResponse;
    private MaterialButton btnSubmit, btnCancel;
    private SimpleRatingBar ratingBar;

    public static ReviewResponseFragment getInstance(LatestReviewModel latestReviewModel) {
        ReviewResponseFragment reviewResponseFragment = new ReviewResponseFragment();
        Bundle bundle = new Bundle();
        bundle.putString("review_response", new Gson().toJson(latestReviewModel));
        reviewResponseFragment.setArguments(bundle);
        return reviewResponseFragment;
    }

    public void setDialogDismissListener(DialogDismissListener dialogDismissListener) {
        this.dialogDismissListener = dialogDismissListener;
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_review_responce_bottomsheet, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        updateViews();
        setListeners();
    }

    protected void setListeners() {
        super.setListeners();
        try {
            etResponse.addTextChangedListener(new TextWatcher() {
                @Override
                public void beforeTextChanged(CharSequence s, int start, int count, int after) {

                }

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {

                }

                @Override
                public void afterTextChanged(Editable s) {
                    tvWordCounter.setText(etResponse.length() + "/300");
                    if (etResponse.length() > 300) {
                        tvWordCounter.setTextColor(getResources().getColor(R.color.red));
                    } else {
                        tvWordCounter.setTextColor(getResources().getColor(R.color.black));
                    }
                }
            });

            btnSubmit.setOnClickListener(v -> {
                if (isValid())
                    addResponse();
            });
            btnCancel.setOnClickListener(v -> dismiss());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private boolean isValid() {
        etResponse.setError(null);

        if (Validators.isNullOrEmpty(etResponse.getText().toString())) {
            etResponse.setError("Please enter your response");
            etResponse.requestFocus();
            return false;
        }
        return true;
    }


    private void addResponse() {
        try {
            HashMap<String, String> params = new HashMap<>();
            params.put("id", reviewModel.id);
            params.put("responce", etResponse.getText().toString());
            params.put("order_id", reviewModel.order_id);
            params.put("customer_id", reviewModel.customer_id);
            params.put("restaurant_id", reviewModel.restaurant_id);

            AndroidNetworking.post(ApiEndPoints.reviews + reviewModel.id)
                    .addQueryParameter(params)
                    .build()
                    .getAsObject(ReviewModel.class, new ParsedRequestListener<ReviewModel>() {
                        @Override
                        public void onResponse(ReviewModel response) {
                            try {
                                if (dialogDismissListener != null) {
                                    dialogDismissListener.onDialogDismiss(etResponse.getText().toString());
                                    dismiss();
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                        @Override
                        public void onError(ANError anError) {
                            anError.printStackTrace();
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void manageIntents() {
        super.manageIntents();
        try {
            if (intentExtrasData.containsKey("review_response")) {
                reviewModel = new Gson().fromJson(String.valueOf(intentExtrasData.get("review_response")), LatestReviewModel.class);
            }
        } catch (JsonSyntaxException e) {
            e.printStackTrace();
        }
    }

    private void updateViews() {
        try {
            if (reviewModel != null) {
                tvName.setText(reviewModel.first_name);
                tvDate.setText(CommonFunctions.formatUnknownDateTime(reviewModel.created, "yyyy-MM-dd", "dd/MM/yyyy"));
                if (!Validators.isNullOrEmpty(reviewModel.rating)) {
                    tvRating.setText(Float.parseFloat(reviewModel.rating) + "/5");
                    ratingBar.setRating(Float.parseFloat(reviewModel.rating));
                } else {
                    tvRating.setText("0/0");
                }

                if (!Validators.isNullOrEmpty(reviewModel.message)) {
                    tvMessage.setVisibility(View.VISIBLE);
                } else {
                    tvMessage.setVisibility(View.GONE);
                }
                tvMessage.setText(reviewModel.message);
                if (!Validators.isNullOrEmpty(reviewModel.responce)) {
                    etResponse.setText(reviewModel.responce);
                    etResponse.setSelection(etResponse.getText().length());
                    tvWordCounter.setText(etResponse.length() + "/300");
                    tvWordCounter.setTextColor(getResources().getColor(R.color.black));
                } else {
                    etResponse.setText("");
                }
            }
        } catch (NumberFormatException e) {
            e.printStackTrace();
        } catch (Resources.NotFoundException e) {
            e.printStackTrace();
        }
    }


    @Override
    protected void initViews(View view) {
        super.initViews(view);
        try {
            ivClose = view.findViewById(R.id.ivClose);
            tvName = view.findViewById(R.id.tvName);
            tvDate = view.findViewById(R.id.tvDate);
            tvRating = view.findViewById(R.id.tvRating);
            ratingBar = view.findViewById(R.id.rating);
            btnSubmit = view.findViewById(R.id.btnSubmit);
            btnCancel = view.findViewById(R.id.btnCancel);
            etResponse = view.findViewById(R.id.etResponse);
            tvMessage = view.findViewById(R.id.tvMessage);
            tvWordCounter = view.findViewById(R.id.tvWordCounter);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onStop() {
        try {
            CommonFunctions.hideKeyboard(getActivity(), etResponse);
        } catch (Exception e) {
            e.printStackTrace();
        }
        super.onStop();
    }
}
