package com.tiffintom.partner1.fragments;

import android.os.Bundle;
import android.text.InputType;
import android.text.TextUtils;
import android.util.Patterns;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ProgressBar;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.widget.AppCompatEditText;
import androidx.fragment.app.DialogFragment;

import com.androidnetworking.AndroidNetworking;
import com.androidnetworking.error.ANError;
import com.androidnetworking.interfaces.ParsedRequestListener;
import com.google.android.material.button.MaterialButton;
import com.google.android.material.tabs.TabLayout;
import com.google.gson.Gson;
import com.tiffintom.partner1.MyApp;
import com.tiffintom.partner1.R;
import com.tiffintom.partner1.common.CommonFunctions;
import com.tiffintom.partner1.common.Validators;
import com.tiffintom.partner1.interfaces.DialogDismissListener;
import com.tiffintom.partner1.models.ApiError;
import com.tiffintom.partner1.models.MerchantCardTransaction;
import com.tiffintom.partner1.network.ApiEndPoints;
import com.tiffintom.partner1.utils.ToastUtils;

/**
 * Created by Amrish on 11-11-2020.
 */
public class SendReceiptFragment extends DialogFragment {
    @Override
    public int getTheme() {
        return R.style.MyDialog;
    }

    @Override
    public void onStart() {
        super.onStart();
        getDialog().getWindow().setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
    }

    private MaterialButton btnConfirm, btnCancel;
    private AppCompatEditText etEmail;
    private TextView tvInfo;
    private DialogDismissListener dialogDismissListener;
    private ProgressBar progressSendReceipt;
    private MyApp myApp = MyApp.getInstance();
    private MerchantCardTransaction merchantCardTransaction;
    private TabLayout tabsLayout;

    public static SendReceiptFragment getInstance(MerchantCardTransaction merchantCardTransaction) {
        SendReceiptFragment rejectReasonDialog = new SendReceiptFragment();
        Bundle bundle = new Bundle();
        bundle.putString("card_object", new Gson().toJson(merchantCardTransaction));
        rejectReasonDialog.setArguments(bundle);
        return rejectReasonDialog;
    }

    public void setDialogDismissListener(DialogDismissListener dialogDismissListener) {
        this.dialogDismissListener = dialogDismissListener;
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.dialog_send_receipt, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        merchantCardTransaction = new Gson().fromJson(getArguments().getString("card_object"), MerchantCardTransaction.class);
        initViews(view);
        setListeners();
    }

    private void initViews(View view) {
        try {
            progressSendReceipt = view.findViewById(R.id.progress_send_receipt);
            btnConfirm = view.findViewById(R.id.btnConfirm);
            btnCancel = view.findViewById(R.id.btnCancel);
            etEmail = view.findViewById(R.id.etIP);
            tvInfo = view.findViewById(R.id.tvInfo);
            tabsLayout = view.findViewById(R.id.tabsLayout);
            tabsLayout.addOnTabSelectedListener(new TabLayout.OnTabSelectedListener() {
                @Override
                public void onTabSelected(TabLayout.Tab tab) {
                    if (tab.getPosition() == 0) {
                        tvInfo.setVisibility(View.VISIBLE);
                        etEmail.setText("");
                        etEmail.setHint(getString(R.string.deliver_to));
                        etEmail.setInputType(InputType.TYPE_TEXT_VARIATION_EMAIL_ADDRESS);
                    } else {
                        tvInfo.setVisibility(View.GONE);
                        etEmail.setText("");
                        etEmail.setHint("Enter your mobile number");
                        etEmail.setInputType(InputType.TYPE_CLASS_PHONE);
                    }
                }

                @Override
                public void onTabUnselected(TabLayout.Tab tab) {

                }

                @Override
                public void onTabReselected(TabLayout.Tab tab) {

                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void setListeners() {
        try {
            etEmail.requestFocus();
            if (!Validators.isNullOrEmpty(merchantCardTransaction.email)) {
                etEmail.setText(merchantCardTransaction.email);
                etEmail.setSelection(etEmail.length());
            }
            btnConfirm.setOnClickListener(view -> {
                etEmail.setError(null);
                if (etEmail.getText().toString().isEmpty()) {
                    etEmail.setError(tabsLayout.getSelectedTabPosition() == 0 ? "Please enter email" : "Please enter mobile number");
                } else {
                    boolean isAllEmailIsValid = true;
                    String[] strings = etEmail.getText().toString().split(",");
                    for (String string : strings) {
                        if (!Validators.isEmail(string)) {
                            isAllEmailIsValid = false;
                        }
                    }
                    if (!isAllEmailIsValid && tabsLayout.getSelectedTabPosition() == 0) {
                        etEmail.setError("Please enter valid email");
                    } else {
                        sendReceipt(merchantCardTransaction);
                    }
                }
            });
            btnCancel.setOnClickListener(view -> {
                if (dialogDismissListener != null)
                    dialogDismissListener.onDialogDismiss(null);
                dismiss();
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void sendReceipt(MerchantCardTransaction merchantTransaction) {
        try {
            progressSendReceipt.setVisibility(View.VISIBLE);
            String fieldName = tabsLayout.getSelectedTabPosition() == 0 ? "email" : "mobile_number";
            AndroidNetworking.post(ApiEndPoints.MERCHANT_SEND_RECEIPT)
                    .addPathParameter("business_id", myApp.getMyPreferences().getBusinessId())
                    .addPathParameter("id", merchantTransaction.id)
                    .addQueryParameter(fieldName, TextUtils.join(",", etEmail.getText().toString().split(",")))
                    .build()
                    .getAsObject(MerchantCardTransaction.class, new ParsedRequestListener() {
                        @Override
                        public void onResponse(Object response) {
                            try {
                                if (response != null) {
                                    ToastUtils.makeLongToast(getActivity(), "Receipt sent successfully");
                                    progressSendReceipt.setVisibility(View.GONE);
                                    if (dialogDismissListener != null)
                                        dialogDismissListener.onDialogDismiss(etEmail.getText().toString());
                                    dismiss();
                                } else {
                                    ToastUtils.makeLongToast(getActivity(), "Receipt could not sent");
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(ANError anError) {
                            try {
                                progressSendReceipt.setVisibility(View.GONE);
                                ApiError apiError = anError.getErrorAsObject(ApiError.class);
                                if (apiError != null && !Validators.isNullOrEmpty(apiError.getMessage())) {
                                    ToastUtils.makeLongToast(getActivity(), apiError.getMessage());
                                } else {
                                    ToastUtils.makeLongToast(getActivity(), "Receipt could not sent");
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
