//
//  SCPRefund.h
//  StripeTerminal
//
//  Created by James Little on 2/20/20.
//  Copyright © 2020 Stripe. All rights reserved.
//
//  Use of this SDK is subject to the Stripe Terminal Terms:
//  https://stripe.com/terminal/legal
//

#import <StripeTerminal/SCPJSONDecodable.h>

NS_ASSUME_NONNULL_BEGIN

/**
 The possible statuses for a refund.

 @see https://stripe.com/docs/api/refunds/object#refund_object-status
 */
typedef NS_ENUM(NSUInteger, SCPRefundStatus) {
    /**
     The refund succeeded.
     */
    SCPRefundStatusSucceeded,

    /**
     The refund is pending.
     */
    SCPRefundStatusPending,

    /**
     The refund failed.
     */
    SCPRefundStatusFailed,

    /**
     Unknown refund state.
     */
    SCPRefundStatusUnknown
} NS_SWIFT_NAME(RefundStatus);

@class SCPPaymentMethodDetails;

/**
 An object representing a Stripe refund.

 Some payment methods, like Interac Debit payments, require that in-person payments
 also be refunded while the cardholder is present. The cardholder must present
 the Interac card to the card reader; these payments cannot be refunded via the
 dashboard or the API.

 @see https://stripe.com/docs/terminal/canada#interac-refunds

 The `collectRefundPaymentMethod:completion` and `confirmRefund` SDK methods
 allow you to build an in-person refund interface into your app.

 The refund SDK methods and the `SCPRefund` class are only available for
 payment methods that require in-person refunds, such as Interac Debit. For
 all other refunds, use the Stripe Dashboard or the Stripe API.

 @see https://stripe.com/docs/api#refunds
 */
NS_SWIFT_NAME(Refund)
@interface SCPRefund : NSObject <SCPJSONDecodable>

/**
 The unique identifier for the refund.
 */
@property (nonatomic, readonly) NSString *stripeId;

/**
 The amount that was refunded.
 */
@property (nonatomic, readonly) NSUInteger amount;

/**
 The ID of the charge that was refunded
 */
@property (nonatomic, readonly) NSString *charge;

/**
 When the refund object was created.
 */
@property (nonatomic, readonly) NSDate *created;

/**
 The currency of the amount refunded.
 */
@property (nonatomic, readonly) NSString *currency;

/**
 Metadata associated with the refund.

 @see https://stripe.com/docs/api#metadata
 */
@property (nonatomic, readonly) NSDictionary<NSString *, NSString *> *metadata;

/**
 Reason for the refund, either user-provided (`duplicate`, `fraudulent`, or
 `requested_by_customer`) or generated by Stripe internally
 (`expired_uncaptured_charge`).
 */
@property (nonatomic, nullable, readonly) NSString *reason;

/**
 The status of the refund.
 */
@property (nonatomic, readonly) SCPRefundStatus status;

/**
 The payment method details associated with the refund.
 */
@property (nonatomic, nullable, readonly) SCPPaymentMethodDetails *paymentMethodDetails;

/**
 If the refund failed, the reason for refund failure if known.
 */
@property (nonatomic, nullable, readonly) NSString *failureReason;

/**
 You cannot directly instantiate this class.
 */
- (instancetype)init NS_UNAVAILABLE;

/**
 You cannot directly instantiate this class.
 */
+ (instancetype)new NS_UNAVAILABLE;

@end

NS_ASSUME_NONNULL_END
